/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.transformation.URIResolver;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.editor.ResourceURIResolver;
import org.ametys.web.explorer.ResourceHelper;

/**
 * {@link URIResolver} for type "explorer-flipbook".<br>
 * These links point to a document file from the resource explorer converted to flash. 
 */
public class Resource2FlipbookUriResolver extends ResourceURIResolver implements PluginAware, Configurable
{
    /** The plugin name. */
    protected String _pluginName;
    
    /** The type. */
    protected String _type;
    
    /** The cache resource component. */
    protected ConvertResource2ImagesComponent _resourceComponent;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resourceComponent = (ConvertResource2ImagesComponent) serviceManager.lookup(ConvertResource2ImagesComponent.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _type = configuration.getChild("type").getValue("explorer-flipbook");
    }
    
    @Override
    public String getType()
    {
        return _type;
    }
    
    @Override
    public String resolve(String uri, boolean download, boolean absolute, boolean internal)
    {
        String path;
        Resource resource = null;
        String siteName = null;
        try
        {
            resource = (Resource) _resolver.resolveById(uri);
            path = getResourcePath(resource);
            siteName = _getSiteName(resource);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Link to unexisting resource " + uri);
            return "";
        }
        
        if (!_resourceComponent.isMimeTypeSupported(resource.getMimeType()))
        {
            return super.resolve(uri, download, absolute, internal);
        }

        StringBuilder result = new StringBuilder();
        
        // _plugins/flipbook/<site>/_[shared-]resource-flipbook/<path>/<to>/<resource>/<fileName>/book.html
        result.append(_getUriPrefix(resource, download, internal, absolute, true))
              .append("/_")
              .append(getRealPrefix(resource, "resource-flipbook"))
              .append(FilenameUtils.encodePath(path))
              .append("/book.html");
        
        new CacheThread(resource.getId(), siteName, _resolver, getLogger()).start();
        
        return URIUtils.encodePath(result.toString());
    }
    
    @Override
    public String resolveImage(String uri, int height, int width, boolean download, boolean absolute, boolean internal)
    {
        Resource resource = null;
        String path;
        try
        {
            resource = (Resource) _resolver.resolveById(uri);
            String fullPath = resource.getPath();
            
            if (fullPath.startsWith(ResourceHelper.SHARED_RESOURCE_PATH))
            {
                path = fullPath.substring(ResourceHelper.SHARED_RESOURCE_PATH.length());
            }
            else
            {
                path = resource.getResourcePath();
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Link to unexisting resource " + uri);
            return "";
        }
        
        if (!_resourceComponent.isMimeTypeSupported(resource.getMimeType()))
        { 
            String defaultPath = _getUriPrefix(resource, download, absolute, internal, true)
                    + "/pages" 
                    + "/error"
                    + "/thumbnail_"
                    + height + "x" + width
                    + ".png";
            
            return URIUtils.encodePath(defaultPath);
        }

        new CacheThread(resource.getId(), _getSiteName(resource), _resolver, getLogger()).start();
        
        String result = _getUriPrefix(resource, download, absolute, internal, false)
                + "/" + getRealPrefix(resource, "resources")
                + FilenameUtils.encodePath(path)
                + "/pages" 
                + "/thumbnail_"
                + height + "x" + width
                + ".png";
        
        return URIUtils.encodePath(result);
    }
    
    @Override
    public String resolveBoundedImage(String uri, int maxHeight, int maxWidth, boolean download, boolean absolute, boolean internal)
    {
        if (maxHeight == 0 && maxWidth == 0)
        {
            return resolve(uri, download, absolute, internal);
        }
        
        Resource resource = null;
        String path;
        try
        {
            resource = (Resource) _resolver.resolveById(uri);
            String fullPath = resource.getPath();
            
            if (fullPath.startsWith(ResourceHelper.SHARED_RESOURCE_PATH))
            {
                path = fullPath.substring(ResourceHelper.SHARED_RESOURCE_PATH.length());
            }
            else
            {
                path = resource.getResourcePath();
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Link to unexisting resource " + uri);
            return "";
        }
        
        if (!_resourceComponent.isMimeTypeSupported(resource.getMimeType()))
        { 
            String defaultPath = _getUriPrefix(resource, download, absolute, internal, true)
                    + "/pages" 
                    + "/error"
                    + "/thumbnail_max"
                    + maxHeight + "x" + maxWidth
                    + ".png";
            
            return URIUtils.encodePath(defaultPath);
        }
        
        new CacheThread(resource.getId(), _getSiteName(resource), _resolver, getLogger()).start();

        String result = _getUriPrefix(resource, download, absolute, internal, false)
                + "/" + getRealPrefix(resource, "resources")
                + FilenameUtils.encodePath(path)
                + "/pages" 
                + "/thumbnail_max"
                + maxHeight + "x" + maxWidth
                + ".png";
        
        return URIUtils.encodePath(result);
    }
    
    private String _getUriPrefix(AmetysObject object, boolean download, boolean absolute, boolean internal, boolean addSiteNameForSharedResources)
    {
        Resource resource = (Resource) object;
        String siteName = _getSiteName(resource);
        
        String uriPrefix = super.getUriPrefix(object, download, absolute, internal);
        
        uriPrefix +=  "/_plugins/" + _pluginName;
        
        if (addSiteNameForSharedResources || !resource.getPath().startsWith(ResourceHelper.SHARED_RESOURCE_PATH))
        {
            uriPrefix += "/" + siteName;
        }

        return uriPrefix;
    }

    private class CacheThread extends Thread 
    {
        private String _resourceId;
        private String _siteName;
        private Logger _cacheLogger;
        private AmetysObjectResolver _cacheResolver;
        
        public CacheThread(String resourceId, String siteName, AmetysObjectResolver cacheResolver, Logger cacheLogger) 
        {
            _resourceId = resourceId;
            _siteName = siteName;
            _cacheLogger = cacheLogger;
            _cacheResolver = cacheResolver;
            setDaemon(true);
            setName("FlipbookCacheResourceCreator");
        }
        
        @Override
        public void run() 
        {
            try
            {
                Resource resource = _cacheResolver.resolveById(_resourceId);
                _resourceComponent.doCache(resource, _siteName);
            }
            catch (Exception e)
            {
                _cacheLogger.error("An error occurred during creating cache for resource : " + _resourceId);
            }
        }
    }
}
