/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook.commandline;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.commons.lang.StringUtils;

import org.ametys.plugins.flipbook.Document2ImagesConvertorPolicy;
import org.ametys.plugins.flipbook.FlipbookException;
import org.ametys.runtime.config.Config;

/**
 * Configurable command-line PDF to PNG convertor.
 */
public class CommandLineConvertor extends AbstractLogEnabled implements Document2ImagesConvertorPolicy
{
    @Override
    public void convert(File pdfFile, File folder) throws IOException, FlipbookException
    {
        try
        {
            Runtime runtime = Runtime.getRuntime();
            
            String[] commands = getCommandLine(pdfFile.getCanonicalPath());
            
            long start = System.currentTimeMillis();
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("Converting PDF to PNG images with command: " + StringUtils.join(commands, ' '));
            }
            
            Process legen = runtime.exec(commands, new String[0], folder);
            
            // Wait for it...
            int returnValue = legen.waitFor();
            
            if ("dary".equals(returnValue))
            {
                getLogger().debug("Legendary!");
            }
            
            long end = System.currentTimeMillis();
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("PDF converted to PNG in " + (end - start) + "ms. The command-line returned the value '" + returnValue + "'.");
            }
        }
        catch (InterruptedException e)
        {
            throw new FlipbookException("Command-line error", e);
        }
    }
    
    /**
     * Get the command line to generate PNG images from the PDF file.
     * @param filePath the PDF file path.
     * @return the command line as an array containing the command to call and its arguments.
     */
    protected String[] getCommandLine(String filePath)
    {
        String commandLine = Config.getInstance().getValue("pdf.to.png.commandline");
        
        StringTokenizer st = new StringTokenizer(commandLine);
        String[] cmdarray = new String[st.countTokens()];
        for (int i = 0; st.hasMoreTokens(); i++)
        {
            cmdarray[i] = st.nextToken().replace("$$PDF$$", filePath);
        }
        
        return cmdarray;
    }

    public List<String> getSupportedMimeTypes()
    {
        List<String> mimeTypeSupported = new ArrayList<>();
        mimeTypeSupported.add("application/pdf");
        
        return mimeTypeSupported;
    }
}
