<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2025 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0" 
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
                xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
                extension-element-prefixes="ametys resolver">
    
    <xsl:import href="workspace:web://stylesheets/variables.xsl"/>
    <xsl:import href="workspace:web://stylesheets/template-default.xsl"/>
    
    <xsl:param name="fileUri"/>
    <xsl:param name="fileName"/>
    
    <xsl:param name="resourceType"/>
    <xsl:param name="resourceId"/>
    
    <xsl:variable name="site-uri-prefix" select="ametys:siteUriPrefix()"/>
    <xsl:variable name="showCover" select="true()"/>
    
    <xsl:template match="/">
        <html>
            <head>
                <meta name="viewport" content="width=device-width, user-scalable=no"/>
                <title>
                    <i18n:translate>
                        <i18n:text i18n:key="PLUGINS_FLIPBOOK_BOOK_TITLE"/>
                        <i18n:param name="filename"><xsl:value-of select="$fileName"/></i18n:param>
                    </i18n:translate>
                </title>
                
                <xsl:call-template name="head-js"/>
                <xsl:call-template name="head-css"/>
            </head>
            <body>
                <xsl:call-template name="body-content"/>
            </body>
        </html>
    </xsl:template>
    
    <xsl:template name="head-css">
        <link rel="stylesheet" href="{ametys:pluginResourceURL('flipbook', 'css/magazine.css')}"/>
        <link rel="stylesheet" href="{ametys:pluginResourceURL('fontawesome6-free', 'css/all.css')}" />
                
        <xsl:call-template name="head-additional-css"/>
    </xsl:template>
    
    <xsl:template name="head-additional-css"/>
    
    <xsl:template name="head-js">
        <!-- JQuery -->
        <xsl:call-template name="head-js-jquery"/>
        
        <script type="text/javascript" src="{$uri-prefix}/plugins/cms/resources/js/AmetysFront.js"></script>
        <script src="{ametys:pluginResourceURL('cms', 'js/AmetysFront/Accessibility.js')}"></script>
        
        <!-- StPageFlip -->
        <script src="{ametys:pluginResourceURL('flipbook', 'js/vendor/page-flip/page-flip.js')}" type="text/javascript"></script>
        <script src="{ametys:pluginResourceURL('flipbook', 'js/magazine.js')}"></script>
        
        <!-- Init flipbook -->        
        <xsl:call-template name="head-js-flipbook"/>
        
        <xsl:call-template name="head-additional-js"/>
    </xsl:template>
    
    <xsl:template name="head-additional-js"/>
    
    <xsl:template name="head-js-flipbook">
        <script>
            // Compute the images dimension according the screen dimension, keeping image ratio and with a max width/height of 80% of the screen
            const widthRatio = screen.width / (<xsl:value-of select="images/image[1]/@width"/>);
            const heightRatio = screen.height / <xsl:value-of select="images/image[1]/@height"/>;
            const minRatio = Math.min(widthRatio, heightRatio)*0.8;

            const imagesPathPrefix = "<xsl:value-of select="concat($site-uri-prefix, '/_plugins/flipbook', images/@pagePath, '/')"/>";

            // Get the max page width and the min page height to determine final dimension of the magazine
            let basePageWidth = 0, basePageHeight = 0, imgWidth, imgHeight;
            
            $j(document).ready(function () {
                <xsl:for-each select="images/image">
                // Image #<xsl:value-of select="position()"/>
                imgWidth = Math.round(<xsl:value-of select="@width"/> * minRatio);
                imgHeight = Math.round(<xsl:value-of select="@height"/> * minRatio);
                
                basePageWidth = Math.max(basePageWidth, imgWidth);
                basePageHeight = basePageHeight == 0 ? imgHeight : Math.min(basePageHeight, imgHeight);
                
                </xsl:for-each>  
                // Create flipbook
                AmetysFlipbook.init('#flipbook', {
                    width: basePageWidth, // base page width
                    height: basePageHeight, // base page height,
                    
                    showCover: <xsl:value-of select="$showCover"/>
                });
            });
        </script>
    </xsl:template>
    
    <xsl:template name="body-content">
        <div class="html-book">
            <xsl:if test="images/image[number(@ratio) > 1]">
                <xsl:attribute name="class">html-book landscape</xsl:attribute>
            </xsl:if>
            <xsl:call-template name="thumbnails"/>
            <xsl:call-template name="top-actions"/>
            <xsl:call-template name="flipbook"/>
            <xsl:call-template name="pagination"/>
            <xsl:call-template name="zoom-result"/>
       </div>
    </xsl:template>
    
    <xsl:template name="zoom-result">
        <div data-flipbook-role="zoom-result" class="zoom-result" role="dialog" aria-modal="true" aria-label="PLUGINS_FLIPBOOK_ZOOM_RESULT_DIALOG_LABEL" i18n:attr="aria-label">
            <div class="zoom-result_container">
                <div class="zoom-result_container_img">
                    <img src="" alt=""/>
                </div>
                <button class="zoom-btn zoom-out" data-flipbook-role="zoom-out" title="PLUGINS_FLIPBOOK_BOOK_ZOOM_OUT" i18n:attr="title"></button>
                <button class="zoom-btn arrow-prev" data-flipbook-role="zoom-prev" title="PLUGINS_FLIPBOOK_NAV_PREV_PAGE" i18n:attr="title"></button>
                <button class="zoom-btn arrow-next" data-flipbook-role="zoom-next" title="PLUGINS_FLIPBOOK_NAV_NEXT_PAGE" i18n:attr="title"></button>
            </div>
        </div>
    </xsl:template>
    
    <xsl:template name="top-actions">
        <div class="top-actions">
            <xsl:call-template name="download-action"/>
            <xsl:call-template name="zoom-action"/>
        </div>
    </xsl:template>
    
    <xsl:template name="download-action">
        <a class="action download" href="{resolver:resolve($resourceType, $resourceId, true())}" title="PLUGINS_FLIPBOOK_BOOK_DOWNLOAD" i18n:attr="title"></a>
    </xsl:template>
    
    <xsl:template name="zoom-action">
        <button class="action zoomin" aria-haspopup="true" data-flipbook-role="zoom-in" title="PLUGINS_FLIPBOOK_BOOK_ZOOM_IN" i18n:attr="title" aria-expanded="false"></button>
    </xsl:template>
    
    <xsl:template name="flipbook">
        <div class="container max">
            <div class="flipbook" id="flipbook">
                <xsl:for-each select="images/image">
                    <div class="page">
                        <div class="page-content">
                            <xsl:variable name="src" select="concat($site-uri-prefix, '/_plugins/flipbook', ../@pagePath, '/', @fileName)"/>
                            <div data-flipbook-page="{position()}" data-flipbook-page-width="{@width}" data-flipbook-page-height="{@height}" class="page-image" style="background-image: url(&quot;{$src}&quot;)">
                            </div>
                        </div>
                    </div>
                </xsl:for-each>    
            </div>
        </div>
    </xsl:template>
    
    <xsl:template name="thumbnails">
        <xsl:param name="size">150</xsl:param>
        
        <div class="thumbnails" data-flipbook-thumbnails="true">
            <div class="thumbnails-inner">
                <nav id="thumbnails" role="navigation" aria-label="plugin.flipbook:PLUGINS_FLIPBOOK_THUMBNAIL_NAVIGATION" i18n:attr="aria-label">
                    <ul>
                        <xsl:for-each select="images/image">
                            <xsl:variable name="name" select="substring-before(@fileName, '.')"/>
                            <xsl:variable name="ext" select="substring-after(@fileName, '.')"/>
                            
                            <li class="thumbnail" data-flipbook-thumbnail="{position()}">
                                <xsl:if test="position() = 1">
                                    <xsl:attribute name="aria-current">true</xsl:attribute>
                                </xsl:if>
                                <button type="button" onclick="AmetysFlipbook.goToPage(this, {position() - 1})">
                                    <xsl:variable name="i18nparams"><page><xsl:value-of select="position()"/></page></xsl:variable> 
                                    <xsl:attribute name="title">
                                        <xsl:value-of select="ametys:translate('plugin.flipbook:PLUGINS_FLIPBOOK_NAV_GO_TO_PAGE', $lang, $i18nparams)"/>
                                    </xsl:attribute>
                                    <img class="thumbnail_img" src="{concat($site-uri-prefix, '/_plugins/flipbook', ../@pagePath, '/', $name, '_max', $size, 'x', $size, '.', $ext)}" alt=""/>
                                </button>
                                <span>
                                    <i18n:translate>
                                        <i18n:text i18n:key="PLUGINS_FLIPBOOK_NAV_PAGE_NUMBER" i18n:catalogue="plugin.flipbook"/>
                                        <i18n:param><xsl:value-of select="position()"/></i18n:param>
                                    </i18n:translate>
                                </span>
                            </li> 
                        </xsl:for-each>  
                    </ul> 
                </nav>
                <xsl:call-template name="thumbnail-btn"/>
            </div>
        </div>
    </xsl:template>
    
    <xsl:template name="thumbnail-btn">
        <button class="toggle-nav" data-flipbook-role="toggle-nav" onclick="AmetysFlipbook.toggleThumbnailNavigation(this)" title="PLUGINS_FLIPBOOK_THUMBNAIL_NAVIGATION_OPEN" i18n:attr="title" aria-expanded="false" aria-controls="thumbnails"></button>
    </xsl:template>
                        
    <xsl:template name="pagination">
        <div class="container">
            <nav class="pagination" role="navigation" aria-label="PLUGINS_FLIPBOOK_PAGINATION" i18n:attr="aria-label">
                <button disabled="true" type="button" class="first" data-flipbook-role="first" title="PLUGINS_FLIPBOOK_NAV_FIRST_PAGE" i18n:attr="title"></button>
                <button disabled="true" type="button" class="prev" data-flipbook-role="prev" title="PLUGINS_FLIPBOOK_NAV_PREV_PAGE" i18n:attr="title"></button>
                <span data-flipbook-role="page-current">1</span> / <span data-flipbook-role="page-total">-</span>
                <button type="button" class="next" data-flipbook-role="next" title="PLUGINS_FLIPBOOK_NAV_NEXT_PAGE" i18n:attr="title"></button>
                <button type="button" class="last" data-flipbook-role="last" title="PLUGINS_FLIPBOOK_NAV_LAST_PAGE" i18n:attr="title"></button>
            </nav>
        </div>
    </xsl:template>
</xsl:stylesheet>
