/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.Strings;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.servlet.RuntimeConfig;

/**
 * Document to images action: converts the document into one image per page if it is not already done.
 */
public abstract class AbstractConvertDocument2ImagesComponent extends AbstractLogEnabled implements ThreadSafe, Initializable, Serviceable, Component
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The document to images convertor. */
    protected Document2ImagesConvertor _documentToImages;
    
    /** The cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    
    /** Map of locks, indexed by resource ID. */
    private ConcurrentMap<String, ReentrantLock> _locks;
    
    @Override
    public void initialize() throws Exception
    {
        _locks = new ConcurrentHashMap<>();
    }   
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _documentToImages = (Document2ImagesConvertor) serviceManager.lookup(Document2ImagesConvertor.ROLE);
    }
    
    /**
     * Test if the mimetype is supported for conversion
     * @param mimeType the mime type
     * @return true if the mimetype can be transformed
     */
    protected boolean isMimeTypeSupported(String mimeType)
    {
        return _documentToImages.getSupportedMimeTypes().contains(mimeType);
    }
    
    /**
     * Create the images and put it into the cache
     * @param relativeCachePath the path of the relative cache
     * @param md5sum the md5 sum
     * @param documentInputStream the input stream of the document
     * @param documentName the document's name
     * @param documentId the id of the document
     * @param documentMimeType the mime type of the document
     * @return The absolute cache path
     * @throws IOException if an error occurs while manipulating files
     * @throws FlipbookException if an error occurs while manipulating the flipbook
     * @throws UnsupportedOperationException If the mime type is not supported
     */
    protected String cache(String relativeCachePath, String md5sum, InputStream documentInputStream, String documentName, String documentId, String documentMimeType) throws IOException, FlipbookException 
    {
        if (!isMimeTypeSupported(documentMimeType))
        {
            throw new UnsupportedOperationException("Cannot convert files of type '" + documentMimeType + "'");
        }
        
        File baseFolder = getCacheFile(relativeCachePath);
        if (!baseFolder.exists())
        {
            baseFolder.mkdirs();
        }
            
        String cachePath = baseFolder.getPath().replace('\\', '/');
        
        File md5File = new File(baseFolder, "document.md5");
        if (!md5File.isFile())
        {
            md5File.createNewFile();
        }
        // Compare the two 
        String oldMd5 = FileUtils.readFileToString(md5File, "UTF-8");
        
        File documentFile = new File(baseFolder, "document/" + documentName);
        
        if (!md5sum.equals(oldMd5) || !documentFile.isFile())
        {
            ReentrantLock lock = new ReentrantLock();
            
            ReentrantLock oldLock = _locks.putIfAbsent(documentId, lock);
            if (oldLock != null)
            {
                lock = oldLock;
            }
            
            lock.lock();
            
            try
            {
                // Test the md5 again, perhaps it was generated by another thread while we were locked.
                oldMd5 = FileUtils.readFileToString(md5File, "UTF-8");
                
                if (!md5sum.equals(oldMd5) || !documentFile.isFile())
                {
                    try
                    {
                        createImages(documentInputStream, documentName, baseFolder);
                    }
                    catch (Throwable t)
                    {
                        getLogger().error("An error occured while converting the document to images \"" + documentFile.getAbsolutePath() + "\"", t);
                        throw new FlipbookException("An error occured while converting the document to images \"" + documentFile.getAbsolutePath() + "\"", t);
                    }
                }
                
                // MD5
                FileUtils.writeStringToFile(md5File, md5sum, "UTF-8");
            }
            finally
            {
                lock.unlock();
                
                if (!lock.hasQueuedThreads())
                {
                    _locks.remove(documentId);
                }
            }
        }
        
        return cachePath;
    }
    
    /**
     * Clean a cache folder if it exists.
     * This method is intended to invalidate cache created by the cache method
     * @param relativeCachePath the relative cache path
     */
    protected void cleanCache(String relativeCachePath)
    {
        File baseFolder = getCacheFile(relativeCachePath);
        if (baseFolder.exists())
        {
            FileUtils.deleteQuietly(baseFolder);
            File parentFolder = baseFolder.getParentFile();
            // if the parent folder exist and is not the flipbook folder, we try do delete it. (it will fail if the folder is not empty)
            while (parentFolder != null && !Strings.CS.equals(parentFolder.getName(), "flipbook") && parentFolder.delete())
            {
                // the folder was empty and is now deleted, let's start again with its parent
                parentFolder = parentFolder.getParentFile();
            }
        }
    }
    
    /**
     * Get the cache file
     * @param relativeCachePath the object path
     * @return the cache file
     */
    public static File getCacheFile(String relativeCachePath)
    {
        return FileUtils.getFile(RuntimeConfig.getInstance().getAmetysHome(), "flipbook", relativeCachePath);
    }

    /**
     * Create images for a resource, in a specified folder.
     * @param documentInputStream the input stream of the document 
     * @param documentName the name of the document
     * @param baseFolder the base folder.
     * @throws IOException if an error occurs while manipulating files
     * @throws FlipbookException if an error occurs while manipulating the flipbook
     */
    protected void createImages(InputStream documentInputStream, String documentName, File baseFolder) throws IOException, FlipbookException
    {
        // Create the document folder.
        File documentFolder = new File(baseFolder, "document");
        documentFolder.mkdirs();
        
        // Create the document file.
        File documentFile = new File(documentFolder, documentName);
        
        try (FileOutputStream fos = new FileOutputStream(documentFile))
        {
            IOUtils.copy(documentInputStream, fos);
            fos.flush();
        }
        
        // Trigger the images creation.
        File imageFolder = new File(baseFolder, "pages");
        imageFolder.mkdirs();
        FileUtils.cleanDirectory(imageFolder);
        
        _documentToImages.convert(documentFile, imageFolder);
    }
}
