/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Objects;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.ExtensionPoint;

/**
 * Converts a document file to PNG images, one image per page.
 * The document images file name must begin with "page".
 * <br>This component just delegates its behavior to the chosen {@link Document2ImagesConvertorPolicy} in configuration <code>flipbook.convertor</code>
 */
public class Document2ImagesConvertor implements Component, Serviceable, Initializable
{
    /** The avalon role. */
    public static final String ROLE = Document2ImagesConvertor.class.getName();
    
    private static final String __CONFIGURATION_NAME = "flipbook.convertor";
    
    /** The {@link ExtensionPoint} for {@link Document2ImagesConvertorPolicy Document2ImagesConvertorPolicies} */
    protected Document2ImagesConvertorPolicyExtensionPoint _document2ImagesConvertorPolicies;
    /** The {@link Document2ImagesConvertorPolicy} chosen in global configuration to use */
    protected Document2ImagesConvertorPolicy _document2ImagesConvertorPolicyInConf;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _document2ImagesConvertorPolicies = (Document2ImagesConvertorPolicyExtensionPoint) manager.lookup(Document2ImagesConvertorPolicyExtensionPoint.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        String document2ImagesConvertorInConfId = Objects.requireNonNull(
                Config.getInstance().getValue(__CONFIGURATION_NAME), 
                String.format("Config param %s is mandatory", __CONFIGURATION_NAME));
        
        _document2ImagesConvertorPolicyInConf = Objects.requireNonNull(
                _document2ImagesConvertorPolicies.getExtension(document2ImagesConvertorInConfId),
                String.format("Extension for id '%s' does not exist for point '%s'", document2ImagesConvertorInConfId, Document2ImagesConvertorPolicyExtensionPoint.ROLE));
    }
    
    /**
     * Get the list of mime types the convertor can convert
     * @return The list of supported mime-types
     */
    public List<String> getSupportedMimeTypes()
    {
        return _document2ImagesConvertorPolicyInConf.getSupportedMimeTypes();
    }
    
    /**
     * Convert the given document file to PNG images, one image per page.
     * @param documentFile the document file to convert.
     * @param folder the destination folder.
     * @throws IOException if an I/O error occurs.
     * @throws FlipbookException if a document error occurs.
     */
    public void convert(File documentFile, File folder) throws IOException, FlipbookException
    {
        _document2ImagesConvertorPolicyInConf.convert(documentFile, folder);
    }
    
}
