/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.imageio.ImageIO;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Generates information on the generated images.
 */
public class ImagesGenerator extends ServiceableGenerator
{
    /** Request attribute from path to images' directory */
    public static final String IMAGES_DIRECTORY_PATH_REQUEST_ATTR = "images-base-directory";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String basePath = (String) request.getAttribute(IMAGES_DIRECTORY_PATH_REQUEST_ATTR);
        
        // Search the document file.
        File documentFolder = new File(basePath, "/document");
        Iterator<File> documentFiles = FileUtils.listFiles(documentFolder, new String[] {"document"}, false).iterator();
        File documentFile = documentFiles.hasNext() ? documentFiles.next() : null;
        
        // List the image files.
        File imageFolder = new File(basePath + "/pages");
        
        List<File> imageFiles = new ArrayList<>(FileUtils.listFiles(imageFolder, new String[] {"png"}, false));
        Collections.sort(imageFiles);
        
        contentHandler.startDocument();
        
        String relPath = URIUtils.encodePath(StringUtils.substringAfter(basePath, "flipbook"));
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("basePath", relPath);
        attrs.addCDATAAttribute("pagePath", relPath + "/pages");
        if (documentFile != null)
        {
            attrs.addCDATAAttribute("documentPath", relPath + "/document");
            attrs.addCDATAAttribute("documentName", documentFile.getName());
            attrs.addCDATAAttribute("documentSize", Long.toString(documentFile.length()));
        }
        
        attrs.addCDATAAttribute("nb-page", String.valueOf(imageFiles.size()));
        
        XMLUtils.startElement(contentHandler, "images", attrs);
        for (File file : imageFiles)
        {
            BufferedImage image = ImageIO.read(file);
            int width = image.getWidth();
            int height = image.getHeight();
            float ratio = (float) width / height;
            
            AttributesImpl pageAttrs = new AttributesImpl();
            pageAttrs.addCDATAAttribute("fileName", file.getName());
            pageAttrs.addCDATAAttribute("width", Integer.toString(width));
            pageAttrs.addCDATAAttribute("height", Integer.toString(height));
            pageAttrs.addCDATAAttribute("ratio", Float.toString(ratio));
            
            XMLUtils.createElement(contentHandler, "image", pageAttrs);
        }
        
        File previewFile = new File(imageFolder, "preview.jpg");
        if (previewFile.exists())
        {
            BufferedImage bi = ImageIO.read(previewFile);
            
            AttributesImpl previewAttrs = new AttributesImpl();
            previewAttrs.addCDATAAttribute("name", previewFile.getName());
            previewAttrs.addCDATAAttribute("width", String.valueOf(bi.getWidth()));
            
            int rows = (int) Math.ceil(1 + imageFiles.size() / 2);
            previewAttrs.addCDATAAttribute("height", String.valueOf(bi.getHeight() / rows));
            XMLUtils.createElement(contentHandler, "preview", previewAttrs);
        }
        
        XMLUtils.endElement(contentHandler, "images");
        
        contentHandler.endDocument();
    }
}
