/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms;

import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.web.cache.AbstractSiteCacheObserver;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.site.Site;

/**
 * {@link Observer} for observing form modification in order to
 * invalidate cache on front-office.
 */
public class InvalidateCacheOnFormModifiedObserver extends AbstractSiteCacheObserver
{
    /** The form DAO */
    protected FormDAO _formDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(FormEvents.FORM_MODIFIED);
    }
    
    @Override
    protected void _internalObserve(Event event, Site site, Session liveSession) throws Exception
    {
        Map<String, Object> args = event.getArguments();
        if (args.containsKey("form"))
        {
            Form form = (Form) args.get("form");
            
            for (ModifiableZoneItem zoneItem : _formDAO.getFormZoneItems(form.getId(), form.getSiteName()))
            {
                SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
                if (sitemapElement instanceof JCRAmetysObject
                    && sitemapElement instanceof ModifiablePage page)
                {
                    JCRAmetysObject jcrPage = (JCRAmetysObject) page;
                    Node pageNode = jcrPage.getNode();
                    
                    if (liveSession.itemExists(pageNode.getPath()))
                    {
                        if (getLogger().isInfoEnabled())
                        {
                            getLogger().info("Form modified: " + event + ", invalidating cache");
                        }
                        
                        _cachePolicy.invalidateCacheOnPageModification(page);
                    }
                }
            }
        }
    }

    @Override
    protected Site _getSite(Event event)
    {
        Map<String, Object> args = event.getArguments();
        if (args.containsKey("form"))
        {
            Form form = (Form) args.get("form");
            return form.getSite();
        }
        
        return null;
    }

}
