/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO.Sort;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.question.types.impl.MatrixQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.DataContext;

/**
 * Get the submitted entries of a form
 *
 */
public class GetFormEntriesAction extends ServiceableAction
{
    /** Constant for whether an entry is in a queue or not */
    public static final String QUEUE_STATUS = "queue-status";
    
    /** The id of the column for entry active or not */
    public static final String FORM_ENTRY_ACTIVE = "active";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    /** The handle limited entries helper */
    protected LimitedEntriesHelper _handleLimitedEntriesHelper;
    
    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    /** The json utils */
    protected JSONUtils _jsonUtils;
    
    /** The I18n utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _formEntryDAO = (FormEntryDAO) smanager.lookup(FormEntryDAO.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _handleLimitedEntriesHelper = (LimitedEntriesHelper) smanager.lookup(LimitedEntriesHelper.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String formId = (String) jsParameters.get("formId");
        Form form = _resolver.resolveById(formId);
        if (form == null)
        {
            throw new ProcessingException("The form of ID '" + formId + " can't be found.");
        }
        
        _formEntryDAO.checkHandleDataRight(form);
        
        Integer offset = _getIntValue(jsParameters, "start", 0);
        Integer limit = _getIntValue(jsParameters, "limit", Integer.MAX_VALUE);

        String sortInfo = (String) jsParameters.get("sort");
        String groupInfo = (String) jsParameters.get("group");
        List<Sort> sorts = _getSorts(formId, sortInfo, groupInfo);
        
        Map<String, Object> matrixLabels = _getMatrixInfos(form);
        
        Map<String, Object> result = new HashMap<>();
        List<Map<String, Object>> entries2json = new ArrayList<>();
        try
        {
            List<FormEntry> entries = _getEntries(form, sorts);
            
            int totalSubmissions = entries.size();
            result.put("total", totalSubmissions);
            
            int currentLimit = 0;
            while (currentLimit < limit && offset < totalSubmissions)
            {
                FormEntry entry = entries.get(offset);
                Model entryModel = (Model) (entry.getModel().toArray())[0];
                Map<String, Object> entryData = new LinkedHashMap<>();
                for (ModelItem modelItem : entryModel.getModelItems())
                {
                    String name = modelItem.getName();
                    FormQuestion question = form.getQuestion(name);
                    if (question != null)
                    {
                        // For question, call specific form API valueToJSONForClient 
                        Object value = question.getType().valueToJSONForClient(entry.getValue(name), question, entry, modelItem);
                        if (value != null)
                        {
                            entryData.put(name, value);
                        }
                    }
                    else
                    {
                        DataContext context = RepositoryDataContext.newInstance()
                                .withObject(entry)
                                .withDataPath(modelItem.getPath());

                        Object value = entry.dataToJSON(name, context);
                        if (value != null)
                        {
                            entryData.put(name, value);
                        }
                    }
                    
                    if (matrixLabels.containsKey(name))
                    {
                        entryData.put(name + "matrice-labels", matrixLabels.get(name));
                    }
                }

                entryData.put(FormEntry.ATTRIBUTE_USER, _userHelper.user2json(entry.getUser(), true));
                
                if (form.isQueueEnabled() && entry.isActive())
                {
                    entryData.put(FormEntry.SYSTEM_ATTRIBUTE_PREFIX + QUEUE_STATUS, _handleLimitedEntriesHelper.isInQueue(entry));
                }
                
                entryData.put(FormEntry.SYSTEM_ATTRIBUTE_PREFIX + FORM_ENTRY_ACTIVE, entry.isActive());
                entryData.put(FormEntry.SYSTEM_ATTRIBUTE_PREFIX + "entryId", entries.get(offset).getId());
                entries2json.add(entryData);
                currentLimit++;
                offset++;
            }
        }
        catch (Exception e)
        {
            getLogger().error("Failed to get entries for form '" + form.getId() + "'.", e);
        }
        
        result.put("entries", entries2json);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }

    /**
     * Get entries and sort
     * @param form the form
     * @param sorts the sorts
     * @return the list of entries
     */
    protected List<FormEntry> _getEntries(Form form, List<Sort> sorts)
    {
        if (sorts.isEmpty())
        {
            return form.getEntries();
        }
        else
        {
            if (sorts.size() == 1 && "user".equals(sorts.get(0).attributeName()))
            {
                String direction = sorts.get(0).direction();
                return form.getEntries()
                    .stream()
                    .sorted((e1, e2) -> "ascending".equals(direction)
                        ? StringUtils.compare(_getUserSortedName(e1), _getUserSortedName(e2))
                        : StringUtils.compare(_getUserSortedName(e2), _getUserSortedName(e1))
                    )
                    .collect(Collectors.toList());
            }
            else
            {
                return _formEntryDAO.getFormEntries(form, false, sorts);
            }
        }
    }
    
    private String _getUserSortedName(FormEntry entry)
    {
        UserIdentity userId = entry.getUser();
        if (userId != null)
        {
            User user = _userManager.getUser(userId);
            return user != null ? user.getSortableName() : null;
        }
        
        return null;
    }
    
    /**
     * Get sorts of search form entry
     * @param formId the form id
     * @param sortString the sort as string
     * @param groupString the group as string
     * @return the list of sort
     */
    protected List<Sort> _getSorts(String formId, String sortString, String groupString)
    {
        List<Sort> sort = new ArrayList<>();
        
        List<Object> sortList = new ArrayList<>(_jsonUtils.convertJsonToList(sortString));
        if (StringUtils.isNotEmpty(groupString))
        {
            // Grouping will be treated server side as a sort. It just needs to be before all the sorters
            sortList.add(0, _jsonUtils.convertJsonToMap(groupString));
        }
        
        for (Object object : sortList)
        {
            if (object instanceof Map)
            {
                Map map = (Map) object;
                String fieldId = (String) map.get("property");
                boolean ascending = "ASC".equals(map.get("direction"));
                
                sort.add(new Sort(
                    StringUtils.contains(fieldId, formId) ? StringUtils.substringAfter(fieldId, formId) : fieldId,
                    ascending ? "ascending" : "descending"
                ));
            }
        }
        
        return sort;
    }
    
    /**
     * Get informations of matrix questions
     * @param form the form
     * @return the map of informations
     */
    protected Map<String, Object> _getMatrixInfos(Form form)
    {
        Map<String, Object> matrixLabels = new HashMap<>();
        List<FormQuestion> matrixQuestions = form.getQuestions()
            .stream()
            .filter(q -> q.getType() instanceof MatrixQuestionType)
            .toList();
        for (FormQuestion matrixQ : matrixQuestions)
        {
            MatrixQuestionType type = (MatrixQuestionType) matrixQ.getType();
            Map<String, String> columns = type.getColumns(matrixQ);
            Map<String, String> rows = type.getRows(matrixQ);
            Map<String, Map<String, String>> matrixInfo = (columns == null || rows == null)
                    ? Map.of()
                    : Map.of(
                        "columns", type.getColumns(matrixQ),
                        "rows", type.getRows(matrixQ)
                      );
            matrixLabels.put(matrixQ.getNameForForm(), matrixInfo);
        }
        return matrixLabels;
    }
    
    private int _getIntValue(Map<String, Object> values, String key, int defaultValue)
    {
        if (values.containsKey(key))
        {
            return Integer.valueOf(values.get(key).toString()).intValue();
        }
        
        return defaultValue;
    }
}
