/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content;

import java.util.List;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.web.repository.content.SharedContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.site.CopyUpdater;

/**
 * Copy updater for contents containing forms
 *
 */
public class FormsCopyUpdater extends AbstractLogEnabled implements CopyUpdater, ThreadSafe, Serviceable
{
    private FormManager _formManager;
    private FormPropertiesManager _formPropertiesManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _formManager = (FormManager) smanager.lookup(FormManager.ROLE);
        _formPropertiesManager = (FormPropertiesManager) smanager.lookup(FormPropertiesManager.ROLE);
    }

    @Override
    public void updateContent(Site initialSite, Site createdSite, Content initialContent, Content createdContent)
    {
        if (createdContent instanceof SharedContent)
        {
            return;
        }
        
        try
        {
            // if the original content has at least one form, we have to update the new content
            boolean needProcess = !_formPropertiesManager.getForms(initialContent).isEmpty();
            
            // before anything, we remove all forms that might exist in the new content copied from initial content
            List<Form> forms = _formPropertiesManager.getForms(createdContent);
            for (Form form : forms)
            {
                _formPropertiesManager.remove(form, createdContent);
                needProcess = true;
            }
            
            if (needProcess)
            {
                // Save form properties and create tables
                _formManager.processContentForms(createdContent);
            }
        }
        catch (Exception e)
        {
            getLogger().warn("[Site copy] Unable to update forms for content '" + createdContent.getId() + "'", e);
        }
    }

    @Override
    public void updatePage(Site initialSite, Site createdSite, Page page)
    {
        // Nothing to do
    }
    
    @Override
    public void updateSite(Site initialSite, Site createdSite)
    {
        // Nothing to do
    }

}
