/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.dao;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.forms.FormEvents;
import org.ametys.plugins.forms.dao.FormEntryDAO.Sort;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper;
import org.ametys.plugins.forms.repository.CopyFormUpdater;
import org.ametys.plugins.forms.repository.CopyFormUpdaterExtensionPoint;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.Form.ExpirationPolicy;
import org.ametys.plugins.forms.repository.FormDirectory;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormFactory;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.parameters.view.ViewParametersManager;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneDAO;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.service.Service;
import org.ametys.web.service.ServiceExtensionPoint;

/**
 * The form DAO
 */
public class FormDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = FormDAO.class.getName();
    /** The right id to handle forms */
    public static final String HANDLE_FORMS_RIGHT_ID = "Plugins_Forms_Right_Handle";

    private static final String __FORM_NAME_PREFIX = "form-";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    /** The form directory DAO */
    protected FormDirectoryDAO _formDirectoryDAO;
    /** The form page DAO */
    protected FormPageDAO _formPageDAO;
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    /** The user helper */
    protected UserHelper _userHelper;
    /** The JCR repository. */
    protected Repository _repository;
    /** The right manager */
    protected RightManager _rightManager;
    /** The service extension point */
    protected ServiceExtensionPoint _serviceEP;
    /** The zone DAO */
    protected ZoneDAO _zoneDAO;
    /** The schedule opening helper */
    protected ScheduleOpeningHelper _scheduleOpeningHelper;
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    /** The site manager */
    protected SiteManager _siteManager;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The copy form updater extension point */
    protected CopyFormUpdaterExtensionPoint _copyFormEP;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _formDirectoryDAO = (FormDirectoryDAO) manager.lookup(FormDirectoryDAO.ROLE);
        _formPageDAO = (FormPageDAO) manager.lookup(FormPageDAO.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _serviceEP = (ServiceExtensionPoint) manager.lookup(ServiceExtensionPoint.ROLE);
        _zoneDAO = (ZoneDAO) manager.lookup(ZoneDAO.ROLE);
        _scheduleOpeningHelper = (ScheduleOpeningHelper) manager.lookup(ScheduleOpeningHelper.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _copyFormEP = (CopyFormUpdaterExtensionPoint) manager.lookup(CopyFormUpdaterExtensionPoint.ROLE);
    }

    /**
     * Check if a user have read rights on a form
     * @param userIdentity the user
     * @param form the form
     * @return true if the user have read rights on a form
     */
    public boolean hasReadRightOnForm(UserIdentity userIdentity, Form form)
    {
        return _rightManager.hasReadAccess(userIdentity, form);
    }
    
    /**
     * Check if a user have write rights on a form element
     * @param userIdentity the user
     * @param formElement the form element
     * @return true if the user have write rights on a form element
     */
    public boolean hasWriteRightOnForm(UserIdentity userIdentity, AmetysObject formElement)
    {
        return _rightManager.hasRight(userIdentity, HANDLE_FORMS_RIGHT_ID, formElement) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Check if a user have write rights on a form
     * @param userIdentity the user
     * @param form the form
     * @return true if the user have write rights on a form
     */
    public boolean hasRightAffectationRightOnForm(UserIdentity userIdentity, Form form)
    {
        return hasWriteRightOnForm(userIdentity, form) || _rightManager.hasRight(userIdentity, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Check rights for a form element as ametys object
     * @param formElement the form element as ametys object
     */
    public void checkHandleFormRight(AmetysObject formElement)
    {
        UserIdentity user = _userProvider.getUser();
        if (!hasWriteRightOnForm(user, formElement))
        {
            throw new AccessDeniedException("User '" + user + "' tried to handle forms without convenient right [" + HANDLE_FORMS_RIGHT_ID + "]");
        }
    }
    
    /**
     * Get all forms from a site
     * @param siteName the site name
     * @return the list of form
     */
    public List<Form> getForms(String siteName)
    {
        String xpathQuery = "//element(" + siteName + ", ametys:site)//element(*, ametys:form)";
        return _resolver.query(xpathQuery)
            .stream()
            .filter(Form.class::isInstance)
            .map(Form.class::cast)
            .collect(Collectors.toList());
    }
    
    /**
     * Get the form properties
     * @param formId The form's id
     * @param full <code>true</code> to get full information on form
     * @param withRights <code>true</code> to have rights in the properties
     * @return The form properties
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getFormProperties (String formId, boolean full, boolean withRights)
    {
        // Assume that no read access is checked (required for bus message target)
        try
        {
            Form form = _resolver.resolveById(formId);
            return getFormProperties(form, full, true);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Can't find form with id: {}. It probably has just been deleted", formId, e);
            Map<String, Object> infos = new HashMap<>();
            infos.put("id", formId);
            return infos;
        }
    }
    
    /**
     * Get the form properties
     * @param form The form
     * @param full <code>true</code> to get full information on form
     * @param withRights <code>true</code> to have rights in the properties
     * @return The form properties
     */
    public Map<String, Object> getFormProperties (Form form, boolean full, boolean withRights)
    {
        Map<String, Object> infos = new HashMap<>();
        
        List<FormEntry> entries = _formEntryDAO.getFormEntries(form, false, List.of(new Sort(FormEntry.ATTRIBUTE_SUBMIT_DATE, "descending")));
        List<SitemapElement> pages = getFormPage(form.getId(), form.getSiteName());
        String workflowName = form.getWorkflowName();

        infos.put("type", "root");
        infos.put("isForm", true);
        infos.put("author", _userHelper.user2json(form.getAuthor(), true));
        infos.put("contributor", _userHelper.user2json(form.getContributor()));
        infos.put("lastModificationDate", DateUtils.zonedDateTimeToString(form.getLastModificationDate()));
        infos.put("creationDate", DateUtils.zonedDateTimeToString(form.getCreationDate()));
        infos.put("entriesAmount", entries.size());
        infos.put("lastEntry", _getLastSubmissionDate(entries));
        infos.put("workflowLabel", StringUtils.isNotBlank(workflowName) ? _workflowHelper.getWorkflowLabel(workflowName) : new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORMS_EDITOR_WORKFLOW_NO_WORKFLOW"));
        
        /** Use in the bus message */
        infos.put("id", form.getId());
        infos.put("name", form.getName());
        infos.put("title", form.getTitle());
        infos.put("fullPath", getFormFullPath(form.getId()));
        infos.put("pages", _getPagesInfos(pages));
        infos.put("hasChildren", form.getPages().size() > 0);
        infos.put("workflowName", workflowName);
        
        infos.put("isConfigured", isFormConfigured(form));
        
        UserIdentity currentUser = _userProvider.getUser();
        if (withRights)
        {
            Set<String> userRights = _getUserRights(form);
            infos.put("rights", userRights);
            infos.put("canEditRight", userRights.contains(HANDLE_FORMS_RIGHT_ID) || _rightManager.hasRight(currentUser, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW);
        }
        else
        {
            boolean canWrite = hasWriteRightOnForm(currentUser, form);
            infos.put("canWrite", canWrite);
            infos.put("canEditRight", canWrite || _rightManager.hasRight(currentUser, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW);
            infos.put("canRead", hasReadRightOnForm(currentUser, form));
        }
        
        if (full)
        {
            infos.put("isPublished", !pages.isEmpty());
            infos.put("hasEntries", !entries.isEmpty());
            infos.put("nbEntries", form.getActiveEntries().size());
            
            FormDirectory formDirectoriesRoot = _formDirectoryDAO.getFormDirectoriesRootNode(form.getSiteName());
            String parentId = form.getParent().getId().equals(formDirectoriesRoot.getId()) ? FormDirectoryDAO.ROOT_FORM_DIRECTORY_ID : form.getParent().getId();
            infos.put("parentId", parentId);
            
            infos.put("isLimitedToOneEntryByUser", form.isLimitedToOneEntryByUser());
            infos.put("isEntriesLimited", form.isEntriesLimited());
            Optional<Long> maxEntries = form.getMaxEntries();
            if (maxEntries.isPresent())
            {
                infos.put("maxEntries", maxEntries.get());
            }
            infos.put("isQueueEnabled", form.isQueueEnabled());
            Optional<Long> queueSize = form.getQueueSize();
            if (queueSize.isPresent())
            {
                infos.put("queueSize", queueSize.get());
            }
            
            infos.put("expirationEnabled", form.isExpirationEnabled());
            
            LocalDate startDate = form.getStartDate();
            LocalDate endDate = form.getEndDate();
            if (startDate != null || endDate != null)
            {
                infos.put("scheduleStatus", _scheduleOpeningHelper.getStatus(form));
                if (startDate != null)
                {
                    infos.put("startDate", DateUtils.localDateToString(startDate));
                }
                if (endDate != null)
                {
                    infos.put("endDate", DateUtils.localDateToString(endDate));
                }
            }

            infos.put("adminEmails", form.hasValue(Form.ADMIN_EMAIL_SUBJECT));
            infos.put("receiptAcknowledgement", form.hasValue(Form.RECEIPT_SENDER));
            infos.put("isAnonymous", _rightManager.hasAnonymousReadAccess(form));
        }
        
        return infos;
    }
    
    /**
     * Get the form title
     * @param formId the form id
     * @return the form title
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public String getFormTitle(String formId)
    {
        Form form = _resolver.resolveById(formId);
        return form.getTitle();
    }
    
    /**
     * Get the form full path
     * @param formId the form id
     * @return the form full path
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public String getFormFullPath(String formId)
    {
        Form form = _resolver.resolveById(formId);
        
        String separator = " > ";
        String fullPath = form.getTitle();
        
        FormDirectory parent = form.getParent();
        if (!_formDirectoryDAO.isRoot(parent))
        {
            fullPath = _formDirectoryDAO.getFormDirectoryPath(parent, separator) + separator + fullPath;
        }
        
        return fullPath;
    }
    
    /**
     * Get user rights for the given form
     * @param form the form
     * @return the set of rights
     */
    protected Set<String> _getUserRights (Form form)
    {
        UserIdentity user = _userProvider.getUser();
        return _rightManager.getUserRights(user, form);
    }
    
    /**
     * Creates a {@link Form}.
     * @param siteName The site name
     * @param parentId The id of the parent.
     * @param name  name The desired name for the new {@link Form}
     * @return The id of the created form
     * @throws Exception if an error occurs during the form creation process
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> createForm (String siteName, String parentId, String name) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        FormDirectory parentDirectory = _formDirectoryDAO.getFormDirectory(siteName, parentId);
        _formDirectoryDAO.checkHandleFormDirectoriesRight(parentDirectory);
        
        String uniqueName = NameHelper.getUniqueAmetysObjectName(parentDirectory, __FORM_NAME_PREFIX + name);
        Form form = parentDirectory.createChild(uniqueName, FormFactory.FORM_NODETYPE);
        
        form.setTitle(name);
        form.setAuthor(_userProvider.getUser());
        form.setCreationDate(ZonedDateTime.now());
        form.setLastModificationDate(ZonedDateTime.now());
        
        parentDirectory.saveChanges();
        String formId = form.getId();
        
        _formPageDAO.createPage(formId, _i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGINS_FORMS_CREATE_PAGE_DEFAULT_NAME")));
        
        result.put("id", formId);
        result.put("name", form.getTitle());
        
        return result;
    }
    
    /**
     * Rename a {@link Form}
     * @param id The id of the form
     * @param newName The new name for the form
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> renameForm (String id, String newName)
    {
        Map<String, String> results = new HashMap<>();
        
        Form form = _resolver.resolveById(id);
        checkHandleFormRight(form);
        
        String uniqueName = NameHelper.getUniqueAmetysObjectName(form.getParent(), __FORM_NAME_PREFIX + newName);
        Node node = form.getNode();
        try
        {
            // Do the move and save it before setting attributes for the {@link LiveWorkspaceListener} 
            // In the other way, attribute are not copied in the live workspace
            node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + uniqueName);
            node.getSession().save();

            form.setTitle(newName);
            form.setContributor(_userProvider.getUser());
            form.setLastModificationDate(ZonedDateTime.now());
            form.saveChanges();
            
            results.put("newName", form.getTitle());
        }
        catch (RepositoryException e)
        {
            getLogger().warn("Form renaming failed.", e);
            results.put("message", "cannot-rename");
        }
        
        results.put("id", id);
        return results;
    }
    
    /**
     * Copies and pastes a form.
     * @param formDirectoryId The id of the form directory target of the copy
     * @param formId The id of the form to copy
     * @return The results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> copyForm(String formDirectoryId, String formId)
    {
        Map<String, String> result = new HashMap<>();
        
        Form originalForm = _resolver.resolveById(formId);
        FormDirectory parentFormDirectory = _resolver.resolveById(formDirectoryId);
        _formDirectoryDAO.checkHandleFormDirectoriesRight(parentFormDirectory);
        
        String uniqueName = NameHelper.getUniqueAmetysObjectName(parentFormDirectory, __FORM_NAME_PREFIX + originalForm.getTitle());
        
        Form cForm = originalForm.copyTo(parentFormDirectory, uniqueName);
        originalForm.copyTo(cForm);
        
        String copyTitle = _i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGIN_FORMS_TREE_COPY_NAME_PREFIX")) + originalForm.getTitle();
        cForm.setTitle(copyTitle);
        cForm.setAuthor(_userProvider.getUser());
        cForm.setCreationDate(ZonedDateTime.now());
        cForm.setLastModificationDate(ZonedDateTime.now());
        cForm.saveChanges();
        
        for (String epId : _copyFormEP.getExtensionsIds())
        {
            CopyFormUpdater copyFormUpdater = _copyFormEP.getExtension(epId);
            copyFormUpdater.updateForm(originalForm, cForm);
        }
        
        result.put("id", cForm.getId());
        
        return result;
    }
    
    /**
     * Deletes a {@link Form}.
     * @param id The id of the form to delete
     * @return The id of the form
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> deleteForm (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        Form form = _resolver.resolveById(id);
        checkHandleFormRight(form);
        
        List<SitemapElement> pages = getFormPage(form.getId(), form.getSiteName());
        if (!pages.isEmpty())
        {
            throw new AccessDeniedException("Can't delete form ('" + form.getId() + "') which contains pages");
        }

        ModifiableAmetysObject parent = form.getParent();
        form.remove();
        parent.saveChanges();
        
        result.put("id", id);
        return result;
    }
    
    /**
     * Moves a {@link Form}
     * @param siteName name of the site
     * @param id The id of the form
     * @param newParentId The id of the new parent directory of the form.
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> moveForm(String siteName, String id, String newParentId)
    {
        Map<String, Object> results = new HashMap<>();
        Form form = _resolver.resolveById(id);
        FormDirectory directory = _formDirectoryDAO.getFormDirectory(siteName, newParentId);
        
        if (hasWriteRightOnForm(_userProvider.getUser(), form) && _formDirectoryDAO.hasWriteRightOnFormDirectory(_userProvider.getUser(), directory))
        {
            _formDirectoryDAO.move(form, siteName, newParentId, results);
        }
        else
        {
            results.put("message", "not-allowed");
        }

        results.put("id", form.getId());
        return results;
    }
    
    /**
     * Change workflow of a {@link Form}
     * @param formId The id of the form
     * @param workflowName The name of new workflow
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> setWorkflow (String formId, String workflowName)
    {
        Map<String, String> results = new HashMap<>();
        
        Form form = _resolver.resolveById(formId);
        checkHandleFormRight(form);
        
        form.setWorkflowName(workflowName);
        form.setContributor(_userProvider.getUser());
        form.setLastModificationDate(ZonedDateTime.now());
        
        form.saveChanges();
        
        results.put("id", formId);

        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", form);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), eventParams));
        
        return results;
    }

    /**
     * Get the submission date of the last entry to the form
     * @param entries A list of form entry ordered by submission date
     * @return the date of the last submission
     */
    protected ZonedDateTime _getLastSubmissionDate(List<FormEntry> entries)
    {
        return entries.isEmpty() ? null : entries.get(0).getSubmitDate();
    }

    /**
     * Get all zone items which contains the form
     * @param formId the form id
     * @param siteName the site name
     * @return the zone items
     */
    public AmetysObjectIterable<ModifiableZoneItem> getFormZoneItems(String formId, String siteName)
    {
        String xpathQuery = "//element(" + siteName + ", ametys:site)//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.forms.service.Display' and ametys:service_parameters/@ametys:formId = '" + formId + "']";
        return _resolver.query(xpathQuery);
    }
    
    /**
     * Get the locale to use for a given form
     * @param form the form
     * @return the locale to use, can be null if the form is not published on a page
     */
    public String getFormLocale(Form form)
    {
        List<SitemapElement> zoneItems = getFormPage(form.getId(), form.getSiteName());
        
        return zoneItems.stream()
                .findFirst()
                .map(SitemapElement::getSitemapName)
                .orElse(null);
    }
    
    /**
     * Get all the page where the form is published
     * @param formId the form id
     * @param siteName the site name
     * @return the list of page
     */
    public List<SitemapElement> getFormPage(String formId, String siteName)
    {
        AmetysObjectIterable<ModifiableZoneItem> zoneItems = getFormZoneItems(formId, siteName);
        
        return zoneItems.stream()
                .map(z -> z.getZone().getSitemapElement())
                .collect(Collectors.toList());
    }
    
    /**
     * Get the page names
     * @param pages the list of page
     * @return the list of page name
     */
    protected List<Map<String, Object>> _getPagesInfos(List<SitemapElement> pages)
    {
        List<Map<String, Object>> pagesInfos = new ArrayList<>();
        for (SitemapElement sitemapElement : pages)
        {
            Map<String, Object> info = new HashMap<>();
            info.put("id", sitemapElement.getId());
            info.put("title", sitemapElement.getTitle());
            info.put("isPage", sitemapElement instanceof Page);
            
            pagesInfos.add(info);
        }
        
        return pagesInfos;
    }
    
    /**
     * Get all the view available for the form display service
     * @param formId the form identifier
     * @param siteName the site name
     * @param language the language
     * @return the views as json
     * @throws Exception if an error occurred
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, Object>> getFormDisplayViews(String formId, String siteName, String language) throws Exception
    {
        List<Map<String, Object>> jsonifiedViews = new ArrayList<>();
    
        Service service = _serviceEP.getExtension("org.ametys.forms.service.Display");
        ElementDefinition viewElementDefinition = (ElementDefinition) service.getParameters().getOrDefault(ViewParametersManager.SERVICE_VIEW_DEFAULT_MODEL_ITEM_NAME, null);
        
        String xpathQuery = "//element(" + siteName + ", ametys:site)/ametys-internal:sitemaps/" + language
                + "//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.forms.service.Display' and ametys:service_parameters/@ametys:formId = '" + formId + "']";
        AmetysObjectIterable<ModifiableZoneItem> zoneItems = _resolver.query(xpathQuery);
        
        Optional<Object> existedServiceView = zoneItems.stream()
            .map(ZoneItem::getServiceParameters)
            .map(sp -> sp.getValue(ViewParametersManager.SERVICE_VIEW_DEFAULT_MODEL_ITEM_NAME))
            .findFirst();
        
        Map<String, I18nizableText> typedEntries = viewElementDefinition.getEnumerator().getEntries();
        for (String id : typedEntries.keySet())
        {
            Map<String, Object> viewAsJson = new HashMap<>();
            viewAsJson.put("id", id);
            viewAsJson.put("label", typedEntries.get(id));
            
            Boolean isServiceView = existedServiceView.map(s -> s.equals(id)).orElse(false);
            if (isServiceView || existedServiceView.isEmpty() && id.equals(viewElementDefinition.getDefaultValue()))
            {
                viewAsJson.put("isDefault", true);
            }
            
            jsonifiedViews.add(viewAsJson);
        }
        
        return jsonifiedViews;
    }
    
    /**
     * <code>true</code> if the form is well configured
     * @param form the form
     * @return <code>true</code> if the form is well configured
     */
    public boolean isFormConfigured(Form form)
    {
        List<FormQuestion> questions = form.getQuestions();
        return !questions.isEmpty() && !questions.stream().anyMatch(q -> !q.getType().isQuestionConfigured(q));
    }
    
    /**
     * Get the dashboard URI
     * @param siteName the site name
     * @return the dashboard URI
     */
    public String getDashboardUri(String siteName)
    {
        String xpathQuery = "//element(" + siteName + ", ametys:site)//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.plugins.forms.workflow.service.dashboard']";
        AmetysObjectIterable<ModifiableZoneItem> zoneItems = _resolver.query(xpathQuery);
        
        Optional<Page> dashboardPage = zoneItems.stream()
                .map(z -> z.getZone().getSitemapElement())
                .filter(Page.class::isInstance)
                .map(Page.class::cast)
                .findAny();
        
        if (dashboardPage.isPresent())
        {
            Page page = dashboardPage.get();
            String pagePath = page.getSitemap().getName() + "/" + page.getPathInSitemap() + ".html";
            
            String url = _siteManager.getSite(siteName).getUrl();
            return url + "/" + pagePath;
        }
        
        return StringUtils.EMPTY;
    }
    
    /**
     * Get the admin dashboard URI
     * @param siteName the site name
     * @return the admin dashboard URI
     */
    public String getAdminDashboardUri(String siteName)
    {
        String xpathQuery = "//element(" + siteName + ", ametys:site)//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.plugins.forms.workflow.service.admin.dashboard']";
        AmetysObjectIterable<ModifiableZoneItem> zoneItems = _resolver.query(xpathQuery);
        
        Optional<Page> dashboardPage = zoneItems.stream()
                .map(z -> z.getZone().getSitemapElement())
                .filter(Page.class::isInstance)
                .map(Page.class::cast)
                .findAny();
        
        if (dashboardPage.isPresent())
        {
            Page page = dashboardPage.get();
            String pagePath = page.getSitemap().getName() + "/" + page.getPathInSitemap() + ".html";
            
            String url = _siteManager.getSite(siteName).getUrl();
            return url + "/" + pagePath;
        }
        
        return StringUtils.EMPTY;
    }

    /**
     * Get the form expiration values
     * @param formId the id of the form
     * @return the values as a JSON map for edition
     */
    @Callable (rights = HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getFormExpiration(String formId)
    {
        Form form = _resolver.resolveById(formId);
        
        return Map.of(Form.EXPIRATION_ENABLED, form.isExpirationEnabled(),
                Form.EXPIRATION_PERIOD, form.getExpirationPeriod(),
                Form.EXPIRATION_POLICY, form.getExpirationPolicy().name());
    }
    
    /**
     * Set the form expiration policy
     * @param formId the id of the form
     * @param expirationPeriod the expiration period in months
     * @param expirationPolicy the expiration policy
     */
    @Callable (rights = HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public void setExpirationPolicy(String formId, long expirationPeriod, String expirationPolicy)
    {
        Form form = _resolver.resolveById(formId);
        
        form.setExpirationPolicy(true, expirationPeriod, ExpirationPolicy.valueOf(expirationPolicy));
        form.saveChanges();
        
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), Map.of("form", form)));
    }
    
    /**
     * Remove the form expiration policy
     * @param formId the id of the form
     */
    public void removeExpirationPolicy(String formId)
    {
        Form form = _resolver.resolveById(formId);
        
        form.setExpirationPolicy(false, -1, null);
        form.saveChanges();
        
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), Map.of("form", form)));
    }
    
}
