/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.enumerators;

import java.util.Map;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * Enumerator for listing values with lazy loading.<p>
 * Such values usually depends on environment (directory listing, DB table, ...).
 * @param <T> Type of the values
 */
public interface LazyEnumerator<T>
{
    /** The pattern parameter key for search */
    public static final String PATTERN_PARAMETER_KEY = "pattern";
    
    /** The sort parameter key for search */
    public static final String SORT_PARAMETER_KEY = "sort";
    
    /** The lang parameter key for search */
    public static final String LANG_PARAMETER_KEY = "lang";
    
    /**
     * Retrieves a single label from a value.
     * @param value the value.
     * @param contextParams the search contextual parameters
     * @return the label or <code>null</code> if not found.
     * @throws Exception if an error occurs.
     */
    public I18nizableText getEntry(T value, Map<String, Object> contextParams) throws Exception;
    
    /**
     * Provides the enumerated values with their optional label.
     * @param contextParams the search contextual parameters
     * @return the enumerated values and their label.
     * @throws Exception if an error occurs.
     */
    public Map<T, I18nizableText> getTypedEntries(Map<String, Object> contextParams) throws Exception;
    
    /**
     * Provides the enumerated values with their optional label.
     * @param contextParams the search contextual parameters
     * @param limit the limit of entries
     * @param paginationData the data to process pagination
     * @return the enumerated values and their label.
     * @throws Exception if an error occurs.
     */
    public Map<T, I18nizableText> searchEntries(Map<String, Object> contextParams, int limit, Object paginationData) throws Exception;
}
