/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The helper to handle admin emails
 */
public class FormAdminMailsHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = FormAdminMailsHelper.class.getName();
    
    /** Pattern for admin mails */
    public static final String REGEX_MAIL = "^([a-zA-Z0-9_\\.\\-])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,})$";
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
    }
    
    /**
     * Get the entries admin emails
     * @param formId Id of the form
     * @return the map of admin emails
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getAdminEmails(String formId)
    {
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        Map<String, Object> emailProperties = new HashMap<> ();
        _addOptionalProperty(Form.ADMIN_EMAILS, form.getAdminEmails(), emailProperties, ArrayUtils.EMPTY_STRING_ARRAY);
        _addOptionalProperty(Form.ADMIN_EMAILS_OTHER, form.getOtherAdminEmails(), emailProperties, StringUtils.EMPTY);
        _addOptionalProperty(Form.ADMIN_EMAIL_LANGUAGE, form.getAdminEmailLanguage(), emailProperties, null);
        _addOptionalProperty(Form.ADMIN_EMAIL_SUBJECT, form.getAdminEmailSubject(), emailProperties, null);
        _addOptionalProperty(Form.ADMIN_EMAIL_BODY, form.getAdminEmailBody(), emailProperties, null);
        
        return emailProperties;
    }
    
    private void _addOptionalProperty(String propertyName, Optional<? extends Object> value, Map<String, Object> limitProperties, Object defaultValue)
    {
        if (value.isPresent())
        {
            limitProperties.put(propertyName, value.get());
        }
        else if (defaultValue != null)
        {
            limitProperties.put(propertyName, defaultValue);
        }
    }
    
    /**
     * Get the entries admin emails
     * @param formId Id of the form
     * @param emails mails of the admin
     * @param receiver name of the form question to get receiver email from, can be null
     * @param subject subject of the admin email
     * @param body body of the admin email
     * @param language The language to use in the admin mail
     * @return the map of results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setAdminEmails(String formId, String emails, String receiver, String subject, String body, String language)
    {
        Map<String, Object> result = new HashMap<>();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        //Check regex
        String[] emailsTab = emails.split("[ ,;\r]");
        List<String> computedEmails = getValidAdminEmails(emailsTab);
        
        if (!computedEmails.isEmpty() || StringUtils.isNotBlank(receiver))
        {
            form.setAdminEmails(computedEmails.toArray(new String[computedEmails.size()]));
            if (StringUtils.isBlank(receiver) && form.hasValue(Form.ADMIN_EMAILS_OTHER))
            {
                form.removeValue(Form.ADMIN_EMAILS_OTHER);
            }
            if (StringUtils.isNotBlank(receiver))
            {
                form.setOtherAdminEmails(receiver);
            }
            
            form.setAdminEmailLanguage(language);
            form.setAdminEmailSubject(subject);
            form.setAdminEmailBody(body);
            form.saveChanges();
        }
        else
        {
            getLogger().error("Mails addresses were undefined or not valid");
            result.put("message", "invalid-address");
        }
        
        return result;
    }

    /**
     * Unset the properties relative to the admin emails
     * @param formId The id ofthe current form
     * @return the map of results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> removeAdminEmails(String formId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        if (form.getAdminEmails().isPresent() || form.getOtherAdminEmails().isPresent())
        {
            if (form.getAdminEmails().isPresent())
            {
                form.removeValue(Form.ADMIN_EMAILS);
            }
            if (form.getOtherAdminEmails().isPresent())
            {
                form.removeValue(Form.ADMIN_EMAILS_OTHER);
            }
            form.removeValue(Form.ADMIN_EMAIL_SUBJECT);
            form.removeValue(Form.ADMIN_EMAIL_BODY);

            form.saveChanges();
        }
        
        return result;
    }
    
    /**
     * Check if an array of emails are valid and returns them without spaces
     * @param emailsTab the emails to check
     * @return the emails trimed or an empty list if they are not valids
     */
    public List<String> getValidAdminEmails(String[] emailsTab)
    {
        Pattern pattern = Pattern.compile(REGEX_MAIL);
        List<String> computedEmails = new ArrayList<>();
        boolean error = false;
        for (String mail : emailsTab)
        {
            String trimMail = mail.trim();
            if (StringUtils.isNotBlank(trimMail))
            {
                if (pattern.matcher(trimMail).matches())
                {
                    computedEmails.add(trimMail);
                }
                else
                {
                    error = true;
                    getLogger().error("Mail address " + mail + " did not match regex");
                }
            }
        }
        return error ? ListUtils.EMPTY_LIST : computedEmails;
    }
}
