/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.validators;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.parameter.DefaultValidator;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * An interval validator for generic type
 * @param <T> the generic type to compare
 */
public abstract class AbstractIntervalFormValidator<T> extends DefaultValidator
{
    /** The min value */
    protected T _min;
    /** The max value */
    protected T _max;
    
    /**
     * The constructor
     * @param regexp the regexp
     * @param mandatory mandatory
     * @param min the min value
     * @param max the max value
     */
    public AbstractIntervalFormValidator (String regexp, boolean mandatory, T min, T max)
    {
        super(regexp, mandatory);
        _min = min;
        _max = max;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected ValidationResult validateSingleValue(Object value)
    {
        ValidationResult result = super.validateSingleValue(value);
        
        if (value != null)
        {
            if (_min != null && isLessThan((T) value, _min))
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("Entry '" + value + "' is invalid : " + _min + " is greater than " + value);
                }
                result.addError(getIntervalError(_min, _max));
            }
            if (_max != null && isMoreThan((T) value, _max))
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("Entry '" + value + "' is invalid : " + _max + " is smaller than " + value);
                }
                result.addError(getIntervalError(_min, _max));
            }
        }
        
        return result;
    }
    
    @Override
    protected ValidationResult validateArrayValues(Object[] values)
    {
        ValidationResult result = super.validateArrayValues(values);
        
        for (Object valueToValidate : values)
        {
            result.addResult(validateSingleValue(valueToValidate));
        }
        
        return result;
    }
    
    /**
     * Test if the first number is less than the second one.
     * @param n1 The first number to compare.
     * @param n2 The second number to compare.
     * @return true if the first number is less than the first, false otherwise.
     */
    protected abstract boolean isLessThan(T n1, T n2);
    
    /**
     * Test if the first number is more than the second one.
     * @param n1 The first number to compare.
     * @param n2 The second number to compare.
     * @return true if the first number is more than the first, false otherwise.
     */
    protected abstract boolean isMoreThan(T n1, T n2);
    
    /**
     * Retrieves the error when the max value is less than the min value
     * @param min The min value
     * @param max The max value
     * @return the interval error message
     */
    protected abstract I18nizableText getIntervalError(T min, T max);
}
