/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The form entry DAO.
 */
 Ext.define('Ametys.plugins.forms.dao.FormEntryDAO', {
 	singleton: true,
 	
 	constructor: function()
 	{
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormEntryDAO
         * @method getFormEntryProperties 
         * Gets properties of a form entry
         * This calls the method 'getFormEntryProperties' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the entry to retrieve
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormEntryDAO",
                methodName: "getFormEntryProperties",
                convertor: this._convertFormEntryProperties,
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_QUESTIONS_GET_PROPERTIES_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
    
 		/**
		 * @callable
		 * @member Ametys.plugins.forms.dao.FormEntryDAO
		 * @method deleteEntry 
		 * delete a form entry
		 * This calls the method 'deleteEntry' of the server DAO 'FormEntryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the entry to retrieve
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.forms.dao.FormEntryDAO",
	     		methodName: "deleteEntry",
	     		callback: {
	         		handler: this._deleteEntryCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_FORMS_ENTRIES_DELETE_ENTRY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		},
	     		localParamsIndex: 1
  			}
  		);
		
 		/**
		 * @callable
		 * @member Ametys.plugins.forms.dao.FormEntryDAO
		 * @method clearEntries 
		 * delete a form entry
		 * This calls the method 'clearEntries' of the server DAO 'FormEntryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the form to retrieve
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.forms.dao.FormEntryDAO",
	     		methodName: "clearEntries",
	     		callback: {
	         		handler: this._clearEntriesCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_FORMS_ENTRIES_CLEAR_ENTRIES_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
 	},
 	
 	/**
 	 * Callback function called after {@link #deleteEntry} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @param {Function} cb The callback
     * @param {Object[]} args The arguments
     * @private
 	 */
 	_deleteEntryCb: function(response, cb, args)
 	{
	    if(response.hasEntries == "false")
	    {
	    	var tool = Ametys.tool.ToolsManager.getTool("uitool-form-entries$" + response.formId);
	    	if(tool)
	    	{
	    		tool.close();
	    	}
	    }
 		Ext.create("Ametys.message.Message", {
	    	type: Ametys.message.Message.DELETED,
	    	targets: {
	    	    id: Ametys.message.MessageTarget.FORM_ENTRY,
	    		parameters: {
					entry: args[1].getParameters().entry
				}
	    	}
	    });
	    Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.FORM_TARGET,
				parameters: {
					id: response.formId
				}
			}
		});
	 },
 	
 	/**
 	 * Callback function called after {@link #deleteEntry} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
	 _clearEntriesCb: function(response)
 	{
 		var tool = Ametys.tool.ToolsManager.getTool("uitool-form-entries$" + response.formId);
    	if(tool)
    	{
    		tool.close();
    	}
    	
 		Ext.create("Ametys.message.Message", {
	    	type: Ametys.message.Message.DELETED,
	    	targets: {
	    	    id: Ametys.message.MessageTarget.FORM_ENTRY,
	    		parameters: {
					entryIds: response.ids,
    				formId: response.formId
				}
	    	}
	    });
	    Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.FORM_TARGET,
				major: true,
				parameters: {
					id: response.formId
				}
			}
		});
	 },
	 
	 /**
     * @private
     * Convert function called after #getFormEntryProperties is processed
     * @param {Object} formEntryProperties The server response
     */
    _convertFormEntryProperties: function (formEntryProperties)
    {
        return Ext.create ('Ametys.plugins.forms.FormEntry', formEntryProperties);
    }
 });
 	