/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.jackrabbit.util.ISO9075;

import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormDirectory;
import org.ametys.plugins.forms.repository.FormDirectoryFactory;
import org.ametys.plugins.forms.repository.FormFactory;
import org.ametys.plugins.forms.repository.FormPage;
import org.ametys.plugins.forms.repository.FormPageFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for manipulating xpath on {@link Form}, {@link FormDirectory} and {@link FormPage}
 */
public class FormXpathUtils extends AbstractLogEnabled implements Component
{
    /**
     * Creates the XPath query to get all form directories
     * @param formDirectory The {@link FormDirectory}, defining the context from which getting children 
     * @return The XPath query
     */
    public static String getXPathForFormDirectories(FormDirectory formDirectory)
    {
        return _getXPathQuery(formDirectory, true, ObjectToReturn.FORM_DIRECTORY);
    }
    
    /**
     * Creates the XPath query to get all form pages
     * @param form The {@link FormDirectory}, defining the context from which getting children 
     * @param onlyDirectChildren <code>true</code> in order to have only direct child forms from parent path, <code>false</code> otherwise to have all forms at any level underneath the parent path
     * @return The XPath query
     */
    public static String getXPathForFormPages(Form form, boolean onlyDirectChildren)
    {
        return _getXPathQuery(form, onlyDirectChildren, ObjectToReturn.FORM_PAGE);
    }
    
    /**
     * Creates the XPath query to get all forms 
     * @param formdirectory The {@link Form}, defining the context from which getting children 
     * @param onlyDirectChildren <code>true</code> in order to have only direct child forms from parent path, <code>false</code> otherwise to have all forms at any level underneath the parent path
     * @return The XPath query
     */
    public static String getXPathForForms(FormDirectory formdirectory, boolean onlyDirectChildren)
    {
        return _getXPathQuery(formdirectory, onlyDirectChildren, ObjectToReturn.FORM);
    }
    
    private static String _getXPathQuery(DefaultTraversableAmetysObject traversableAO, boolean onlyDirectChildren, ObjectToReturn objectToReturn)
    {
        StringBuilder parentPath = _getParentPath(traversableAO);
        final String slashOrDoubleSlash = onlyDirectChildren ? "/" : "//";
        StringBuilder query = parentPath
                .append(slashOrDoubleSlash)
                .append("element(*, ")
                .append(objectToReturn.toNodetype())
                .append(")");
               
        return query.toString();
    }
    
    private static StringBuilder _getParentPath(DefaultTraversableAmetysObject traversableAO)
    {
        try
        {
            StringBuilder parentPath = new StringBuilder("/jcr:root")
                    .append(ISO9075.encodePath(traversableAO.getNode().getPath()));
            return parentPath;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    private static enum ObjectToReturn
    {
        FORM,
        FORM_PAGE,
        FORM_DIRECTORY;
        
        private String _nodetype;
        
        static
        {
            FORM._nodetype = FormFactory.FORM_NODETYPE;
            FORM_DIRECTORY._nodetype = FormDirectoryFactory.FORM_DIRECTORY_NODETYPE;
            FORM_PAGE._nodetype = FormPageFactory.FORM_PAGE_NODETYPE;
        }
        
        String toNodetype()
        {
            return _nodetype;
        }
    }
}
