/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.cache;

import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.AllowedUsers;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.service.StaticService;

/**
 * Class representing a business service. <br>
 * A service is identified by an id and a Cocoon-URL.<br>
 * This URL corresponds to a pipeline called by a page template.<br>
 * URL must be relative to the sitemap of the plugin containing the service.
 */
public class FormService extends StaticService
{

    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    /** The right manager */
    protected RightManager _rightManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }

    @Override
    public boolean isCacheable(Page currentPage, ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String formId = serviceParameters.getValue("formId");
        Form form = (Form) _resolver.resolveById(formId);

        // Both conditions must be fulfilled to be cacheable
        return form.isCacheable() && _isLessRestrictive(form, currentPage);
    }
    
    /**
     *  Compare the form rights to the content rights : As both are independent, we can have a page were not every user can see the form
     *  The form is cacheable only if form rights are less restrictive, so everyone seeing this page can see the form
     * @param form the form
     * @param page the page
     * @return true if the forms right are less restrictive than the page using it
     */
    private boolean _isLessRestrictive(Form form, Page page)
    {    
        if (_rightManager.hasAnonymousReadAccess(form))
        {
            return true;
        }
        if (_rightManager.hasAnonymousReadAccess(page))
        {
            return false;
        }
        if (_rightManager.hasAnyConnectedUserReadAccess(form))
        {
            return true;
        }
        if (_rightManager.hasAnyConnectedUserReadAccess(page))
        {
            return false;
        }
        
        AllowedUsers formAllowedUsers = _rightManager.getReadAccessAllowedUsers(form);
        AllowedUsers pageAllowedUsers = _rightManager.getReadAccessAllowedUsers(page);
        
        Set<UserIdentity> formSet = formAllowedUsers.resolveAllowedUsers(true);
        Set<UserIdentity> pageSet = pageAllowedUsers.resolveAllowedUsers(true);
        
        return formSet.containsAll(pageSet);
    }
}
