/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Class representing a form.
 */
public class Form
{
    /** The form id. */
    protected String _id;
    /** The form label. */
    protected String _label;
    /** The field which indicates the email to send an acknowledgement of receipt. */
    protected String _receiptFieldId;
    /** The email address of the sender. Can be empty to use default one. */
    protected String _receiptFieldFromAddress;
    /** The subject og the email. Cannot be empty */
    protected String _receiptFieldSubject;
    /** The body of the email. Cannot be empty */
    protected String _receiptFieldBody;
    /** The id of the page where to redirect to. Can be empty */
    protected String _redirectTo;
    /** The form emails. */
    protected Set<String> _notificationEmails;
    /** The limit of entries */
    protected String _limit;
    /** The message to display when they are remaining places */
    protected String _remainingPlaces;
    /** The message to display when they are no remaining places */
    protected String _noRemainingPlaces;
    
    /** The form content ID. */
//    protected String _contentId;
    
    /** The form fields. */
    protected List<Field> _fields;
    
    /** The form fieldsets. */
    protected List<Fieldset> _fieldsets;
    
    /** The name of the form's workflow */
    protected String _workflowName;
    
    /** The content id */
    protected String _contentId;

    /**
     * Default constructor.
     */
    public Form()
    {
        this("", "", "", "", "", "", new HashSet<>(), "", new ArrayList<>(), new ArrayList<>(), "", "", "", "", "");
    }
    
    /**
     * Constructor with parameters.
     * @param id the form ID.
     * @param label the form label.
     * @param receiptFieldId the acknowledgement of receipt field ID.
     * @param receiptFieldFromAddress The sender address for the receipt mail. Can be empty
     * @param receiptFieldSubject The receipt mail subject
     * @param receiptFieldBody The receipt mail body
     * @param emails the form emails.
     * @param contentId the form content ID.
     * @param fields the form fields.
     * @param fieldsets the form fieldsets.
     * @param redirectTo the id of the page where to redirect to
     * @param workflowName the name of the workflow
     * @param limit the limit of entries
     * @param remainingPlaces the remainingPlaces
     * @param noRemainingPlaces the noRemainingPlaces
     */
    public Form(String id, String label, String receiptFieldId, String receiptFieldFromAddress, String receiptFieldSubject, String receiptFieldBody, Set<String> emails, String contentId, Collection<Field> fields, Collection<Fieldset> fieldsets, String redirectTo, String workflowName, String limit, String remainingPlaces, String noRemainingPlaces)
    {
        this._id = id;
        this._label = label;
        this._receiptFieldId = receiptFieldId;
        this._receiptFieldFromAddress = receiptFieldFromAddress;
        this._receiptFieldSubject = receiptFieldSubject;
        this._receiptFieldBody = receiptFieldBody;
        this._notificationEmails = emails;
        this._contentId = contentId;
        this._fields = new ArrayList<>(fields);
        this._fieldsets = new ArrayList<>(fieldsets);
        this._redirectTo = redirectTo;
        this._workflowName = workflowName;
        this._limit = limit;
        this._remainingPlaces = remainingPlaces;
        this._noRemainingPlaces = noRemainingPlaces;
    }

    /**
     * Get the id.
     * @return the id
     */
    public String getId()
    {
        return _id;
    }

    /**
     * Set the id.
     * @param id the id to set
     */
    public void setId(String id)
    {
        this._id = id;
    }

    /**
     * Get the label.
     * @return the label
     */
    public String getLabel()
    {
        return _label;
    }

    /**
     * Set the label.
     * @param label the label to set
     */
    public void setLabel(String label)
    {
        this._label = label;
    }

    /**
     * Get the acknowledgement of receipt field ID.
     * @return the acknowledgement of receipt field ID.
     */
    public String getReceiptFieldId()
    {
        return _receiptFieldId;
    }
    
    /**
     * Set the acknowledgement of receipt field ID.
     * @param receiptFieldId the acknowledgement of receipt field ID.
     */
    public void setReceiptFieldId(String receiptFieldId)
    {
        this._receiptFieldId = receiptFieldId;
    }
    
    /**
     * Get the receiptFieldFromAddress
     * @return the receiptFieldFromAddress
     */
    public String getReceiptFieldFromAddress()
    {
        return _receiptFieldFromAddress;
    }
    
    /**
     * Set the receiptFieldFromAddress
     * @param receiptFieldFromAddress the receiptFieldFromAddress to set
     */
    public void setReceiptFieldFromAddress(String receiptFieldFromAddress)
    {
        _receiptFieldFromAddress = receiptFieldFromAddress;
    }
    
    /**
     * Get the receiptFieldBody
     * @return the receiptFieldBody
     */
    public String getReceiptFieldBody()
    {
        return _receiptFieldBody;
    }
    
    /**
     * Set the receiptFieldBody
     * @param receiptFieldBody the receiptFieldBody to set
     */
    public void setReceiptFieldBody(String receiptFieldBody)
    {
        _receiptFieldBody = receiptFieldBody;
    }
    
    /**
     * Get the receiptFieldSubject
     * @return the receiptFieldSubject
     */
    public String getReceiptFieldSubject()
    {
        return _receiptFieldSubject;
    }
    
    /**
     * Set the receiptFieldSubject
     * @param receiptFieldSubject the receiptFieldSubject to set
     */
    public void setReceiptFieldSubject(String receiptFieldSubject)
    {
        _receiptFieldSubject = receiptFieldSubject;
    }
    
    /**
     * Get the emails.
     * @return the emails
     */
    public Set<String> getNotificationEmails()
    {
        return _notificationEmails;
    }

    /**
     * Set the emails.
     * @param emails the emails to set
     */
    public void setNotificationEmails(Set<String> emails)
    {
        this._notificationEmails = emails;
    }
    
    /**
     * Get the content ID.
     * @return the content ID.
     */
    public String getContentId()
    {
        return _contentId;
    }

    /**
     * Set the content ID.
     * @param contentId the content ID to set.
     */
    public void setContentId(String contentId)
    {
        this._contentId = contentId;
    }
    
    /**
     * Get the fields.
     * @return the fields
     */
    public List<Field> getFields()
    {
        return _fields;
    }
    
    /**
     * Set the fields.
     * @param fields the fields to set
     */
    public void setFields(Collection<Field> fields)
    {
        _fields = new ArrayList<>(fields);
    }
    
    /**
     * Get a copy of the form's fields, indexed by its ID.
     * @return a copy of the form's fields, indexed by its ID.
     */
    public Map<String, Field> getFieldMap()
    {
        Map<String, Field> fieldMap = new LinkedHashMap<>();
        for (Field field : _fields)
        {
            fieldMap.put(field.getId(), field);
        }
        return fieldMap;
    }
    
    /**
     * Get the fieldsets.
     * @return the fieldsets
     */
    public List<Fieldset> getFieldsets()
    {
        return _fieldsets;
    }
    
    /**
     * Set the fieldsets.
     * @param fieldsets the fieldsets to set
     */
    public void setFieldsets(Collection<Fieldset> fieldsets)
    {
        this._fieldsets = new ArrayList<>(fieldsets);
    }

    /**
     * the redirectTo
     * @return the redirectTo
     */
    public String getRedirectTo()
    {
        return _redirectTo;
    }
    
    /**
     * the redirectTo
     * @param redirectTo the redirectTo to set
     */
    public void setRedirectTo(String redirectTo)
    {
        _redirectTo = redirectTo;
    }
    
    /**
     * Retrieve the name of the workflow of this form's entries
     * @return the name of the workflow used
     */
    public String getWorkflowName()
    {
        return _workflowName;
    }

    /**
     * Set the name of the workflow of this form's entries
     * @param workflowName the name of the workflow to use
     */
    public void setWorkflowName(String workflowName)
    {
        _workflowName = workflowName;
    }

    /**
     * the limit of entries in the form
     * @return the limit of entries
     */
    public String getLimit()
    {
        return _limit;
    }

    /**
     * Set the limit of entries in the form
     * @param limit the limit of entries
     */
    public void setLimit(String limit)
    {
        this._limit = limit;
    }

    /**
     * Get the message to display when they are remaining places
     * @return the remainingPlaces
     */
    public String getRemainingPlaces()
    {
        return _remainingPlaces;
    }

    /**
     * Set the message to display when they are remaining places
     * @param remainingPlaces the remainingPlaces to set
     */
    public void setRemainingPlaces(String remainingPlaces)
    {
        this._remainingPlaces = remainingPlaces;
    }

    /**
     * Get the message to display when they are no remaining places
     * @return the remainingPlaces
     */
    public String getNoRemainingPlaces()
    {
        return _noRemainingPlaces;
    }

    /**
     * Set the message to display when they are no remaining places
     * @param noRemainingPlaces the noRemainingPlaces to set
     */
    public void setNoRemainingPlaces(String noRemainingPlaces)
    {
        this._noRemainingPlaces = noRemainingPlaces;
    }
}
