/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.User;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.repository.RepositoryConstants;

/**
 * Generator listing all of the forms that are no longer referenced by any content
 */
public class GetOrphanFormsAction extends ServiceableAction
{
    /** Component gathering methods to query forms in JCR */
    private FormPropertiesManager _formPropertiesManager;
    
    /** Component gathering methods to manipulate SQL forms */
    private FormTableManager _formTableManager;
    
    /** The users manager. */
    private UserManager _userManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) serviceManager.lookup(FormTableManager.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<> ();
        try
        {
            String formId = null;
            Form form = null;
            
            List<String> tableNames = _formTableManager.getTableNames();
            
            List<Map<String, Object>> tableList = new ArrayList<> ();
            for (String tableName : tableNames)
            {
                Map<String, Object> table = new HashMap<> ();
                
                formId = tableName.substring(FormTableManager.TABLE_PREFIX.length());
                if (formId.length() == 0)
                {
                    getLogger().error("Find a SQL form table with a empty id '" + tableName + "'. It will be ignored. This table should be deleted manually.");
                }
                else
                {
                    form = _formPropertiesManager.getForm(formId);
                    if (form == null)
                    {
                        Node frozenContentNode = _formPropertiesManager.getMostRecentFormFrozenContent(formId);
                        if (frozenContentNode != null)
                        {
                            String contentUUId = frozenContentNode.getProperty(JcrConstants.JCR_FROZENUUID).getString();
                            String title = frozenContentNode.getProperty("ametys:title").getString();
                            
                            table.put("contentTitle", title);
                            
                            if (frozenContentNode.hasProperty(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":site"))
                            {
                                table.put("siteName", frozenContentNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":site").getString());
                            }
                            
                            Node contributorNode = frozenContentNode.getNode(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":contributor"); 
                            
                            String contributorLogin = contributorNode.getProperty("ametys:login").getString();
                            User user = _userManager.getUser(contributorNode.getProperty("ametys:population").getString(), contributorNode.getProperty("ametys:login").getString());
                            table.put("lastContributor", user != null ? user.getFullName() : contributorLogin);
                            
                            boolean stillExists = true;
                            try
                            {
                                frozenContentNode.getSession().getNodeByIdentifier(contentUUId);
                            }
                            catch (ItemNotFoundException e)
                            {
                                stillExists = false;
                            }
                            
                            table.put("stillExists", stillExists);
                        }
                        
                        // Table attrs
                        table.put("name", tableName);
                        table.put("total", Integer.toString(_formTableManager.getTotalSubmissions(formId)));
                        tableList.add(table);
                    }
                }
            }
            result.put("tables", tableList);
        }
        catch (Exception e)
        {
            throw new ProcessingException("An error occurred while retrieving the orphan forms.", e);
        }
        
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
}
