/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.processing;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.pageaccess.IsSitemapRestrictedAction;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinsManager;

/**
 * Get a content.
 */
public class GetContentAction extends org.ametys.cms.content.GetContentAction
{
    private SiteManager _siteManager;
    private SkinsManager _skinsManager;
    private URIPrefixHandler _uriPrefixHandler;
    private RightManager _rightManager;
    private CurrentUserProvider _currentUserProvied;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _skinsManager = (SkinsManager) serviceManager.lookup(SkinsManager.ROLE);
        _uriPrefixHandler = (URIPrefixHandler) manager.lookup(URIPrefixHandler.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvied = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public Map<String, String> act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = super.act(redirector, resolver, objectModel, source, parameters);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String pluginName = (String) request.getAttribute("pluginName");
        String url = "";
        
        Content content = (Content) request.getAttribute(Content.class.getName());
        if (content instanceof WebContent)
        {
            WebContent webContent = (WebContent) content;
            
            // Protecting the form
            if (!_rightManager.currentUserHasReadAccess(webContent))
            {
                UserIdentity user = _currentUserProvied.getUser();
                if (user == null)
                {
                    throw new AuthorizationRequiredException();
                }
                else
                {
                    throw new AccessDeniedException("User '" + UserIdentity.userIdentityToString(user) + "' can not post form for content '" + webContent.getId() + "'");
                }
            }
            
            // The user is granted, let's mark the request to ensure that IsSitemapRestrictedAction will not block (since the root autorization required can be stronger)  FORMS-330
            request.setAttribute(IsSitemapRestrictedAction.REQUEST_ATTRIBUTE_DONOTCHECKRIGHTS, true);
            
            String siteName = webContent.getSiteName();
            String lang = webContent.getLanguage();
            String template = null;
            
            result.put("site", siteName);
            result.put("lang", lang);
            
            // Take the HEAD revision
            String revision = ((VersionableAmetysObject) content).getRevision();
            ((VersionableAmetysObject) content).switchToRevision(null);
            
            Iterator<Page> pages = webContent.getReferencingPages().iterator();
            if (pages.hasNext())
            {
                template = pages.next().getTemplate();
            }
            else
            {
                // We certainly come from a virtual page, try to get template from referer
                Site site = _siteManager.getSite(siteName);
                Sitemap sitemap = site.getSitemap(lang);
                
                String absoluteUriPrefix = _uriPrefixHandler.getAbsoluteUriPrefix(siteName) + "/" + lang + "/";
                
                String referer = request.getHeader("Referer");
                if (referer != null && referer.startsWith(absoluteUriPrefix))
                {
                    String pathInSitemap = referer.substring(absoluteUriPrefix.length());
                    try
                    {
                        Page page = sitemap.getChild(pathInSitemap.substring(0, pathInSitemap.indexOf(".html")));
                        
                        // Get template from referer page
                        template = page.getTemplate();
                    }
                    catch (UnknownAmetysObjectException e)
                    {
                        // No page found, get default template
                        template = _getDefaultTemplate(site);
                    }
                }
                else
                {
                    template = _getDefaultTemplate(site);
                }
            }
            
            url = "cocoon://" + siteName + "/" + lang + "/_plugins/" + pluginName + "/" + template + "/process-form";
            result.put("template", template);
            result.put("url", url);
            
            ((VersionableAmetysObject) content).switchToRevision(revision);
        }
        else
        {
            url = "cocoon:/process-form";
            result.put("url", url);
        }
        
        return result;
    }
    
    private String _getDefaultTemplate (Site site)
    {
        String skinName = site.getSkinId();
        
        Skin skin = _skinsManager.getSkin(skinName);
        
        Set<String> templates = skin.getTemplates();
        
        if (templates.contains("page"))
        {
            // Get template 'page' if exists
            return "page";
        }
        
        // Get the first template
        return templates.iterator().next();
    }

}
