/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;

/**
 * The helper for acknowledgement of receipts
 */
public class AcknowledgementOfReceiptsHelper implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = AcknowledgementOfReceiptsHelper.class.getName();
   
    /** Pattern for senders mail */
    public static final String REGEX_SENDER_MAIL = "^([a-zA-Z0-9_\\.\\-])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,})$";

    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
    }
    
    /**
     * Get the form properties relative to the acknowledgement of receipt
     * @param formId The id of the form
     * @return  a map of the form acknowledgement of receipt properties 
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getAcknowledgementReceiptProperties(String formId)
    {
        Map<String, Object> receiptProperties = new HashMap<> ();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        _addOptionalProperty(Form.RECEIPT_SENDER, form.getReceiptSender(), receiptProperties, Config.getInstance().getValue("smtp.mail.from"));
        _addOptionalProperty(Form.RECEIPT_RECEIVER, form.getReceiptReceiver(), receiptProperties, FormMailHelper.RECEIVER_COMBOBOX_ENTRY_USER_VALUE);
        _addOptionalProperty(Form.RECEIPT_SUBJECT, form.getReceiptSubject(), receiptProperties, null);
        _addOptionalProperty(Form.RECEIPT_BODY, form.getReceiptBody(), receiptProperties, null);
        
        return receiptProperties;
    }
    
    private void _addOptionalProperty(String propertyName, Optional<? extends Object> value, Map<String, Object> limitProperties, Object defaultValue)
    {
        if (value.isPresent())
        {
            limitProperties.put(propertyName, value.get());
        }
        else if (defaultValue != null)
        {
            limitProperties.put(propertyName, defaultValue);
        }
    }
    
    /**
     * Set the properties relative to the acknowledgement of receipt 
     * @param formId id of the form
     * @param sender the sender email address
     * @param receiver name for form of question where the receiver's email address will be entered
     * @param subject The subject for the email 
     * @param body The body of the email
     * @return the map of results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setAcknowledgementReceiptProperties(String formId, String sender, String receiver, String subject, String body)
    {
        Map<String, Object> result = new HashMap<>();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        Pattern pattern = Pattern.compile(REGEX_SENDER_MAIL);
        Matcher matcher = pattern.matcher(sender);
        if (matcher.matches())
        {
            form.setReceiptSender(sender);
            form.setReceiptReceiver(receiver);
            form.setReceiptSubject(subject);
            form.setReceiptBody(body);
            
            form.saveChanges();
        }
        
        return result;
    }
    
    /**
     * Unset the properties relative to the acknowledgement of receipt 
     * @param formId The id ofthe current form
     * @return the map of results
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> removeEntriesReceipt(String formId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        if (form.getReceiptReceiver().isPresent())
        {
            form.removeValue(Form.RECEIPT_SENDER);
            form.removeValue(Form.RECEIPT_RECEIVER);
            form.removeValue(Form.RECEIPT_SUBJECT);
            form.removeValue(Form.RECEIPT_BODY);

            form.saveChanges();
        }
        
        return result;
    }
}
