/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Helper for manipulating workflows for {@link Form}
 */
public class FormWorkflowHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = FormWorkflowHelper.class.getName();
    
    /** The edit action type */
    public static final String EDIT_ACTION = "edit";
    /** The edit by submitter action type */
    public static final String EDIT_BY_SUBMITTER_ACTION = "edit-by-submitter";
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _workflowProvider = (WorkflowProvider) serviceManager.lookup(WorkflowProvider.ROLE);
        _workflowHelper = (WorkflowHelper) serviceManager.lookup(WorkflowHelper.ROLE);
    }
    
    /**
     * Initialize workflow for form entry
     * @param entry the form entry
     * @throws Exception if an error occurred
     */
    public void initializeWorkflow(FormEntry entry) throws Exception
    {
        Form form = entry.getForm();
        String workflowName = form.getWorkflowName();
        if (StringUtils.isNotBlank(workflowName))
        {
            // create workflow with entry id
            Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
            
            int initialActionId = _workflowHelper.getInitialAction(workflowName);
            
            Map<String, Object> inputs = new HashMap<>();
            inputs.put(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY, entry);
            
            long workflowInstanceId = workflow.initialize(workflowName, initialActionId, inputs);
            entry.setWorkflowId(workflowInstanceId);
        }
    }
    
    /**
     * Get the available actions for the given form entry and wanted action types
     * @param entry The form entry
     * @param types The action types to include. If empty, all available actions are returned. Otherwise, only actions with the specified types are returned.
     * @return The list of wanted available actions
     */
    public List<ActionDescriptor> getAvailableActionsRestrictedToTypes(FormEntry entry, List<String> types)
    {
        Predicate<String> includeTest = actionType -> {
            return types.isEmpty() || types.contains(actionType);
        };
        
        return getAvailableActions(entry, includeTest);
    }
    
    /**
     * Get the non excluded available actions for the given form entry
     * @param entry The form entry
     * @param excludedTypes The action types to exclude. If empty, all available actions are returned. Otherwise, only actions whose type is <b>not</b> listed are returned.
     * @return The list of available actions except the excluded ones
     */
    public List<ActionDescriptor> getAvailableActionsExcludingTypes(FormEntry entry, List<String> excludedTypes)
    {
        Predicate<String> includeTest = actionType -> {
            return excludedTypes.isEmpty() || !excludedTypes.contains(actionType);
        };
        
        return getAvailableActions(entry, includeTest);
    }
    
    /**
     * Get the wanted available actions for the given form entry
     * @param entry The form entry
     * @param actionTypeFilter The filter to apply on the action type or null to retrieve all actions
     * @return The list of accepted available actions
     */
    protected List<ActionDescriptor> getAvailableActions(FormEntry entry, Predicate<String> actionTypeFilter)
    {
        List<ActionDescriptor> availableActions = new ArrayList<>();

        Map<String, Object> inputs = new HashMap<>();
        inputs.put(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY, entry);
     
        Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
        int[] actions = workflow.getAvailableActions(entry.getWorkflowId(), inputs);
        
        WorkflowDescriptor workflowDescriptor = _workflowHelper.getWorkflowDescriptor(entry.getForm().getWorkflowName());
        for (int actionId : actions)
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            Map<String, Object> metaAttributes = action.getMetaAttributes();
            String actionType = (String) metaAttributes.getOrDefault("action-type", StringUtils.EMPTY);
            
            if (actionTypeFilter == null || actionTypeFilter.test(actionType))
            {
                availableActions.add(action);
            }
        }
        
        return availableActions;
    }
}
