/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.autofill;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/**
 * Class for using common User attributes as default value
 */
public class CurrentUserAutoFillSource extends AbstractStaticAutoFillSource
{
    /** Constant for autofill attribute. */
    public static final String ATTRIBUTE_AUTOFILL = "autofill";
    
    /** Name of email autofillStaticEnumerator entry */
    public static final String EMAIL_AUTOFILL_VALUE = "email";
    
    /** Name of id autofillStaticEnumerator entry */
    public static final String ID_AUTOFILL_VALUE = "id";
    
    /** Name of fullname autofillStaticEnumerator entry */
    public static final String FULLNAME_AUTOFILL_VALUE = "fullName";
    
    /** Name of firstName autofillStaticEnumerator entry */
    public static final String FIRSTNAME_AUTOFILL_VALUE = "firstName";
    
    /** Name of lastName autofillStaticEnumerator entry */
    public static final String LASTNAME_AUTOFILL_VALUE = "lastName";
    

    /** Map of ModelItems specific to DefaultStaticAutoFillSource */
    protected Map<String, ModelItem> _defaultAutoFillItems;
    
    public String getAutofillValue(FormQuestion question)
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
        if (user != null)
        {
            String autofill = question.getValue(ATTRIBUTE_AUTOFILL);
            switch (autofill)
            {
                case "email":
                    return user.getEmail();
                case "id":
                    return user.getIdentity().getLogin();
                case "fullName":
                    return user.getFullName();
                case "firstName":
                    return user.getFirstName();
                case "lastName":
                    return user.getLastName();
                default:
                    return null;
            }
        }
        return null;
    }

    public Map<String, ModelItem> getModelItems()
    {
        _defaultAutoFillItems = new HashMap<>();
        ElementDefinition<String> autofill = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_AUTOFILL, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_DESC", new DefaultValidator(null, true));

        StaticEnumerator<String> autofillStaticEnumerator = new StaticEnumerator<>();
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_EMAIL"), EMAIL_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_ID"), ID_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_FULLNAME"), FULLNAME_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_FIRSTNAME"), FIRSTNAME_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_LASTNAME"), LASTNAME_AUTOFILL_VALUE);
        autofill.setEnumerator(autofillStaticEnumerator);
        autofill.setDefaultValue(EMAIL_AUTOFILL_VALUE);
        
        _defaultAutoFillItems.put(ATTRIBUTE_AUTOFILL, autofill);
        
        return _defaultAutoFillItems;
    }

    public List<ViewItem> getViewElements()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        ViewElement autofill = new ViewElement();
        autofill.setDefinition((ElementDefinition< ? >) _defaultAutoFillItems.get(ATTRIBUTE_AUTOFILL));
        viewElements.add(autofill);
        
        return viewElements;
    }

    public boolean isCacheable(FormQuestion question)
    {
        return false;
    }

}
