/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.sources;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for creating choice list with costs from manual source
 */
public class ManualWithCostsSourceType extends AbstractSourceType
{
    /** Constant for element definition name of the select grid*/
    public static final String ATTRIBUTE_GRID_COSTS = "select-grid-costs";
    
    /** Map of ModelItems specific to ManualWithCostsSourceType */
    protected Map<String, ModelItem> _manualWithCostsChoicesItems;
    
    public Map<String, ModelItem> getModelItems()
    {
        _manualWithCostsChoicesItems = new HashMap<>();
        
        ElementDefinition<String> grid = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_GRID_COSTS, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_GRID", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_GRID_DESC", null);
        grid.setWidget("manual-source-grid");
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("displayCost", new I18nizableText("true"));
        grid.setWidgetParameters(widgetParameters);
        _manualWithCostsChoicesItems.put(grid.getName(), grid);
       
        return _manualWithCostsChoicesItems;
    }
    
    public List<ViewItem> getViewItems()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        ViewElement grid = new ViewElement();
        grid.setDefinition((ElementDefinition< ? >) _manualWithCostsChoicesItems.get(ATTRIBUTE_GRID_COSTS));
        viewElements.add(grid);
        
        return viewElements;
    }

    @Override
    public List<String> getFieldToDisableIfFormPublished()
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished();
        fieldNames.add(ATTRIBUTE_GRID_COSTS);
        return fieldNames;
    }
    
    public boolean remoteData()
    {
        return false;
    }

    public I18nizableText getEntry(ChoiceOption value, Map<String, Object> contextParams) throws Exception
    {
        Map<ChoiceOption, I18nizableText> typedEntries = getTypedEntries(contextParams);
        return typedEntries.get(value);
    }

    public Map<ChoiceOption, I18nizableText> getTypedEntries(Map<String, Object> contextParams) throws Exception
    {
        Map<ChoiceOption, I18nizableText> entries = new LinkedHashMap<>();
        
        FormQuestion question = _getQuestionFromParam(contextParams);
        String gridOptionAsJson = question.getValue(ATTRIBUTE_GRID_COSTS);
        Map<String, Object> options = _jsonUtils.convertJsonToMap(gridOptionAsJson);
        for (Entry<String, Object> entry : options.entrySet())
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> values = (Map<String, Object>) entry.getValue();
            String label = entry.getKey();
            Double cost = ((Number) values.get("cost")).doubleValue();
            ChoiceOptionWithCost choice = new ChoiceOptionWithCost(values.get("value"), cost);
            entries.put(choice, new I18nizableText(label));
        }
                
        return entries;
    }
    
    /**
     * Get the entry cost from the value
     * @param value the value
     * @param contextParams the search contextual parameters
     * @return the cost of the entry
     * @throws Exception if an error occurred
     */
    public Double getEntryCost(String value, Map<String, Object> contextParams) throws Exception
    {
        Map<ChoiceOption, I18nizableText> typedEntries = getTypedEntries(contextParams);
        return typedEntries.keySet()
            .stream()
            .filter(ChoiceOptionWithCost.class::isInstance)
            .map(ChoiceOptionWithCost.class::cast)
            .filter(o -> o.getValue().equals(value))
            .map(o -> o.getCost())
            .findAny()
            .orElse(null);
    }

    public Map<ChoiceOption, I18nizableText> searchEntries(Map<String, Object> contextParams, int limit, Object paginationData) throws Exception
    {
        throw new UnsupportedOperationException("Method searchEntries can't be called for ManualWithCostsSourceType");
    }
    
    public Object getAnonimizedValue(FormQuestion question)
    {
        String gridOptionAsJson = question.getValue(ATTRIBUTE_GRID_COSTS);
        Map<String, Object> options = _jsonUtils.convertJsonToMap(gridOptionAsJson);
        return options.values().stream().findFirst()
            .map(v -> ((Map<String, Object>) v).get("value"))
            .orElse(null);
    }
}
