/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.holder.DataHolderRelativeDisableCondition;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditions;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditionsHelper;
import org.ametys.plugins.forms.question.computing.ComputingType;
import org.ametys.plugins.forms.question.computing.ComputingTypeExtensionPoint;
import org.ametys.plugins.forms.question.types.AbstractStaticFormQuestionType;
import org.ametys.plugins.forms.question.types.ConfidentialAwareQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for creating computed question
 */
public class ComputedQuestionType extends AbstractStaticFormQuestionType implements ConfidentialAwareQuestionType
{
    /** Constant for computing attribute. */
    public static final String ATTRIBUTE_COMPUTING_TYPE = "computing-type";

    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_COMPUTED";
    
    /** The computing type extension point */
    private ComputingTypeExtensionPoint _computingTypeExtensionPoint;
    
    /** The data holder relative disable condition helper */
    private DataHolderRelativeDisableConditionsHelper _disableConditionsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _computingTypeExtensionPoint = (ComputingTypeExtensionPoint) manager.lookup(ComputingTypeExtensionPoint.ROLE);
        _disableConditionsHelper = (DataHolderRelativeDisableConditionsHelper) manager.lookup(DataHolderRelativeDisableConditionsHelper.ROLE);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        ElementDefinition<String> computingType = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_COMPUTING_TYPE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTED_COMPUTING_TYPE", "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTED_COMPUTING_TYPE_DESC", null);
        StaticEnumerator<String> typesStaticEnumerator = new StaticEnumerator<>();
        for (ComputingType type : _getAllComputingType())
        {
            typesStaticEnumerator.add(type.getLabel(), type.getId());
            
            DisableCondition condition = new DataHolderRelativeDisableCondition(ATTRIBUTE_COMPUTING_TYPE, OPERATOR.NEQ, type.getId(), true, _disableConditionsHelper);
            
            Map<String, ModelItem> typeModelItems = type.getModelItems();
            for (ModelItem item : typeModelItems.values())
            {
                DisableConditions itemDisableConditions = item.getDisableConditions() != null ? item.getDisableConditions() : new DataHolderRelativeDisableConditions();
                itemDisableConditions.getConditions().add(condition);
                item.setDisableConditions(itemDisableConditions);
            }
            modelItems.addAll(typeModelItems.values());
        }
        computingType.setEnumerator(typesStaticEnumerator);
        modelItems.add(computingType);
        
        modelItems.add(getConfidentialityModelItem());
        
        return modelItems;
    }
    
    @Override
    protected List<ViewItemGroup> _getTabs(Form form)
    {
        return List.of(
                _getMainTab(),
                _getAdvancedTab(),
                _getRulesTab()
                );
    }
    
    /**
     * Define the content of the main tab
     * @return the main tab definition
     */
    protected SimpleViewItemGroup _getMainTab()
    {
        SimpleViewItemGroup fieldset =  super._createMainTab();
        
        ViewElement computingCombobox = new ViewElement();
        computingCombobox.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_COMPUTING_TYPE));
        fieldset.addViewItem(computingCombobox);

        for (ComputingType type : _getAllComputingType())
        {
            if (!type.getViewElements().isEmpty())
            {
                fieldset.addViewItems(type.getViewElements());
            }
        }
        
        return fieldset;
    }
    
    /**
     * Define the content of the advanced tab
     * @return the advanced tab definition
     */
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._createAdvancedTab();
        
        ViewElement title = new ViewElement();
        title.setDefinition((ElementDefinition< ? >) getModel().getModelItem(FormQuestion.ATTRIBUTE_TITLE));
        advancedFieldset.addViewItem(title);
        
        ViewElement nameForForms = new ViewElement();
        nameForForms.setDefinition((ElementDefinition< ? >) getModel().getModelItem(FormQuestion.ATTRIBUTE_NAME_FOR_FORM));
        advancedFieldset.addViewItem(nameForForms);
        
        advancedFieldset.addViewItem(getConfidentialityViewElement(getModel()));
        
        return advancedFieldset;
    }
    
    public String getStorageType(FormQuestion question)
    {
        ComputingType computingType = getComputingType(question);
        if (computingType != null)
        {
            return computingType.getStorageType(question);
        }
        
        // Return string by default but it not supposed to have no computing type
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        super.saxAdditionalInfos(contentHandler, question);
        for (ComputingType type : _getAllComputingType())
        {
            AttributesImpl attr = new AttributesImpl();
            String xslEnabled = StringUtils.isBlank(type.getXSLT())  ? "false" : "true";
            attr.addCDATAAttribute("xsl-enabled", xslEnabled);
            attr.addCDATAAttribute("id", type.getId());
            XMLUtils.createElement(contentHandler, "computing-type", attr);
        }
    }
    
    /**
     * Get all computing types
     * @return a list of {@link ComputingType}
     */
    private List<ComputingType> _getAllComputingType()
    {
        List<ComputingType> computingTypes = new ArrayList<>();
        for (String id: _computingTypeExtensionPoint.getExtensionsIds())
        {
            computingTypes.add(_computingTypeExtensionPoint.getExtension(id));
        }
        return computingTypes;
    }
    
    @Override
    public boolean isQuestionConfigured(FormQuestion question)
    {
        return StringUtils.isNotBlank(question.getValue(ATTRIBUTE_COMPUTING_TYPE));
    }
    
    /**
     * Get the computing type
     * @param question the question
     * @return the computing type
     */
    public ComputingType getComputingType(FormQuestion question)
    {
        String value = question.getValue(ATTRIBUTE_COMPUTING_TYPE);
        return StringUtils.isNotBlank(value) ? _computingTypeExtensionPoint.getExtension(value) : null;
    }
    
    @Override
    public Set<String> getDisplayXSLTs()
    {
        Set<String> displayXSLTs = new HashSet<>(super.getDisplayXSLTs());
        _getAllComputingType().stream()
            .map(ComputingType::getXSLT)
            .filter(StringUtils::isNotBlank)
            .forEach(displayXSLTs::add);
        
        return displayXSLTs;

    }
    
    @Override
    public boolean onlyForDisplay(FormQuestion question)
    {
        ComputingType computingType = getComputingType(question);
        if (computingType != null)
        {
            return !computingType.hasComputedValue();
        }
        return true;
    }
    
    @Override
    public boolean canBeAnsweredByUser(FormQuestion question)
    {
        return false;
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public void saxEntryValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        super.saxEntryValue(contentHandler, question, entry);
        AttributesImpl attrs = new AttributesImpl();
        ComputingType computingType = getComputingType(question);
        attrs.addCDATAAttribute("computed-type", computingType.getId());
        
        XMLUtils.startElement(contentHandler, "additional-infos", attrs);

        computingType.saxAdditionalValue(contentHandler, question, entry);
        
        XMLUtils.endElement(contentHandler, "additional-infos");
        
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_COMPUTING_TYPE);
        
        ComputingType computingType = getComputingType(question);
        if (computingType != null)
        {
            for (String fieldName : computingType.getFieldToDisableIfFormPublished())
            {
                fieldNames.add(fieldName);
            }
        }
        
        return fieldNames;
    }

    @Override
    public String getJSRenderer(FormQuestion question)
    {
        ComputingType computingType = getComputingType(question);
        return computingType.getJSRenderer();
    }
    
    @Override
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        Object valueToJSONForClient = super.valueToJSONForClient(value, question, entry, modelItem);
        ComputingType computingType = getComputingType(question);
        return computingType.valueToJSONForClient(valueToJSONForClient, question, entry, modelItem);
    }

    @Override
    public boolean isCacheable(FormQuestion question)
    {
        return getComputingType(question) == null || getComputingType(question).isCacheable();
    }
    
    public ModelItem getConfidentialityModelItem()
    {
        return _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_CONFIDENTIALITY, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_CONFIDENTIAL", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_CONFIDENTIAL_DESC", null);
    }
    
    public ViewItem getConfidentialityViewElement(Model model)
    {
        ViewElement confidential = new ViewElement();
        confidential.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_CONFIDENTIALITY));
        
        return confidential;
    }

    public boolean isConfidential(FormQuestion formQuestion)
    {
        return formQuestion.getValue(ATTRIBUTE_CONFIDENTIALITY, false, false);
    }
    
    public Map<String, Object> getAnonymizedData(FormQuestion question)
    {
        ComputingType computingType = getComputingType(question);
        return computingType.getAnonymizedData(question);
    }
}
