/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.holder.DataHolderRelativeDisableCondition;
import org.ametys.cms.data.holder.DataHolderRelativeDisableConditions;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.question.autofill.AutofillSource;
import org.ametys.plugins.forms.question.autofill.AutofillSourceExtensionPoint;
import org.ametys.plugins.forms.question.types.AbstractFormQuestionType;
import org.ametys.plugins.forms.question.types.AutocompleteAwareQuestionType;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/**
 * Class for creating simple text questions
 */
public class SimpleTextQuestionType extends AbstractFormQuestionType implements AutocompleteAwareQuestionType
{
    /** Constant for regexp attribute. */
    public static final String ATTRIBUTE_REGEXP = "regexp";
    
    /** Constant for custom regexp attribute. */
    public static final String ATTRIBUTE_CUSTOM_REGEX = "custom-regex";
    
    /** Constant for autofill source attribute. */
    public static final String ATTRIBUTE_AUTOFILL_SOURCE = "autofill-source";
    
    /** Name of empty regexStaticEnumerator entry  */
    public static final String EMPTY_REGEX_VALUE = " ";
    
    /** Name of email regexStaticEnumerator entry  */
    public static final String EMAIL_REGEX_VALUE = "email";
    
    /** Name of phone regexStaticEnumerator entry  */
    public static final String PHONE_REGEX_VALUE = "phone";
    
    /** Name of regexStaticEnumerator entry that enable custom regex field */
    public static final String CUSTOM_REGEX_VALUE = "custom";
    
    /** Constant for placeholder attribute. */
    public static final String ATTRIBUTE_PLACEHOLDER = "placeholder";

    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_SIMPLE_TEXT";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The users manager */
    protected UserManager _userManager;
    
    /** The autofill source extension point */
    protected AutofillSourceExtensionPoint _autoFillSourceExtensionPoint;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _autoFillSourceExtensionPoint = (AutofillSourceExtensionPoint) manager.lookup(AutofillSourceExtensionPoint.ROLE);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        //PLACEHOLDER
        ElementDefinition placeholder = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PLACEHOLDER, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_PLACEHOLDER", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_PLACEHOLDER_DESC", null);
        modelItems.add(placeholder);
        
        //AUTOFILL SOURCE
        ElementDefinition<String> autofillSource = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_AUTOFILL_SOURCE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_SOURCE", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_SOURCE_DESC", null);
        StaticEnumerator<String> sourcesStaticEnumerator = new StaticEnumerator<>();
        sourcesStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_EMPTY_TEXT"), StringUtils.EMPTY);
        for (AutofillSource source : _getAllAutoFillSources())
        {
            sourcesStaticEnumerator.add(source.getLabel(), source.getId());
            
            DisableCondition condition = new DataHolderRelativeDisableCondition(ATTRIBUTE_AUTOFILL_SOURCE, OPERATOR.NEQ, source.getId(), true, _disableConditionsHelper);
            
            Map<String, ModelItem> sourceModelItems = source.getModelItems();
            for (ModelItem item : sourceModelItems.values())
            {
                DisableConditions itemDisableConditions = item.getDisableConditions() != null ? item.getDisableConditions() : new DataHolderRelativeDisableConditions();
                itemDisableConditions.getConditions().add(condition);
                item.setDisableConditions(itemDisableConditions);
            }
            modelItems.addAll(sourceModelItems.values());
        }
        autofillSource.setEnumerator(sourcesStaticEnumerator);
        autofillSource.setDefaultValue(StringUtils.EMPTY);
        modelItems.add(autofillSource);
        
        //REGEX
        ElementDefinition<String> regexp = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_REGEXP, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_DESC", null);
        
        StaticEnumerator<String> regexStaticEnumerator = new StaticEnumerator<>();
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_EMPTY_TEXT"), EMPTY_REGEX_VALUE);
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_EMAIL"), EMAIL_REGEX_VALUE);
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_PHONE"), PHONE_REGEX_VALUE);
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_CUSTOM"), CUSTOM_REGEX_VALUE);
        regexp.setEnumerator(regexStaticEnumerator);
        Map<String, I18nizableText> regexpWidgetParameters  = new HashMap<>();
        regexpWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        regexp.setWidgetParameters(regexpWidgetParameters);
        regexp.setDefaultValue(EMPTY_REGEX_VALUE);
        modelItems.add(regexp);
        
        //CUSTOM REGEX
        ElementDefinition customRegex = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_CUSTOM_REGEX, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_CUSTOM", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_CUSTOM_DESC", new DefaultValidator("^\\/.+\\/[gi]*$", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_REGEXP_ERROR_DESCRIPTION"), false));
        DisableConditions regexDisableConditions = new DataHolderRelativeDisableConditions();
        DisableCondition regexCondition = new DataHolderRelativeDisableCondition(ATTRIBUTE_REGEXP, OPERATOR.NEQ, CUSTOM_REGEX_VALUE, _disableConditionsHelper);
        regexDisableConditions.getConditions().add(regexCondition);
        customRegex.setDisableConditions(regexDisableConditions);
        modelItems.add(customRegex);
        
        //AUTOCOMPLETE
        modelItems.add(getAutocompleteModelItem(InputType.TEXT));
        
        return modelItems;
    }
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._getAdvancedTab();
        
        ViewElement placeholder = new ViewElement();
        placeholder.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_PLACEHOLDER));
        advancedFieldset.addViewItem(placeholder);
        
        ViewElement regexp = new ViewElement();
        regexp.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_REGEXP));
        advancedFieldset.addViewItem(regexp);
        
        ViewElement customRegex = new ViewElement();
        customRegex.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_CUSTOM_REGEX));
        advancedFieldset.addViewItem(customRegex);
        
        advancedFieldset.addViewItem(getAutocompleteViewElement(getModel()));
       
        ViewElement autofill = new ViewElement();
        autofill.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_AUTOFILL_SOURCE));
        advancedFieldset.addViewItem(autofill);
        
        for (AutofillSource source : _getAllAutoFillSources())
        {
            if (!source.getViewElements().isEmpty())
            {
                advancedFieldset.addViewItems(source.getViewElements());
            }
        }
        
        return advancedFieldset;
    }
    
    public String getStorageType(FormQuestion question)
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        ModelItem item = super._getEntryModelItem(question);
        ((ElementDefinition) item).setValidator(new DefaultValidator(_getEntryRegExpPattern(question), isMandatory(question)));
        return item;
    }

    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        super.saxAdditionalInfos(contentHandler, question);
        
        if (StringUtils.isNotBlank(question.getValue(ATTRIBUTE_AUTOFILL_SOURCE)))
        {
            _getAutoFillSource(question).saxAdditionalInfos(contentHandler, question);
        }
        
        String pattern = _getRegExpPattern(question);
        if (StringUtils.isNotEmpty(pattern))
        {
            XMLUtils.createElement(contentHandler, "pattern", pattern);
        }
    }
    
    /**
     * Get the autofill source
     * @param question the question
     * @return the autofill source
     */
    private AutofillSource _getAutoFillSource(FormQuestion question)
    {
        String value = question.getValue(ATTRIBUTE_AUTOFILL_SOURCE);
        return StringUtils.isNotBlank(value) ? _autoFillSourceExtensionPoint.getExtension(value) : null;
    }
    
    /**
     * Get all autofill sources
     * @return a list of {@link AutofillSource}
     */
    private List<AutofillSource> _getAllAutoFillSources()
    {
        List<AutofillSource> autoFillSources = new ArrayList<>();
        for (String id: _autoFillSourceExtensionPoint.getExtensionsIds())
        {
            autoFillSources.add(_autoFillSourceExtensionPoint.getExtension(id));
        }
        return autoFillSources;
    }
    
    /**
     * Get the validation pattern.
     * @param question the question
     * @return the validation pattern.
     */
    protected String _getRegExpPattern (FormQuestion question)
    {
        String regexpType = question.getValue(ATTRIBUTE_REGEXP);
        if (regexpType == null)
        {
            return null;
        }
        else if ("email".equals(regexpType))
        {
            return "/^([a-zA-Z0-9_\\.\\-\\+])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,4})+$/";
        }
        else if ("phone".equals(regexpType))
        {
            return "/^(\\+?\\(?[0-9]{1,3}\\)?([\\s]?)(\\(0\\))?|0)([\\s]?)([0-9\\-\\+\\s]{4,})+$/";
        }
        else if ("custom".equals(regexpType) && !("".equals(question.getValue(ATTRIBUTE_CUSTOM_REGEX))))
        {
            return question.getValue(ATTRIBUTE_CUSTOM_REGEX);
        }
        
        return null;
    }
    
    /**
     * Get the validation pattern for entries.
     * @param question the question
     * @return the validation pattern.
     */
    protected String _getEntryRegExpPattern (FormQuestion question)
    {
        String regexpType = question.getValue(ATTRIBUTE_REGEXP);
        if (regexpType == null)
        {
            return null;
        }
        else if ("email".equals(regexpType))
        {
            return "^([a-zA-Z0-9_\\.\\-\\+])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,4})+$";
        }
        else if ("phone".equals(regexpType))
        {
            return "^(\\+?\\(?[0-9]{1,3}\\)?([\\s]?)(\\(0\\))?|0)([\\s]?)([0-9\\-\\+\\s]{4,})+$";
        }
        else if ("custom".equals(regexpType) && !("".equals(question.getValue(ATTRIBUTE_CUSTOM_REGEX))))
        {
            String custom = question.getValue(ATTRIBUTE_CUSTOM_REGEX);
            return custom.substring(1, custom.length() - 1);
        }
        
        return null;
    }
    
    @Override
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        super.validateQuestionValues(values, errors);
        
        if (StringUtils.isNotBlank((String) values.get(ATTRIBUTE_CUSTOM_REGEX))
                && !Pattern.matches("^\\/.+\\/[gi]*$", (CharSequence) values.get(ATTRIBUTE_CUSTOM_REGEX)))
        {
            errors.put(ATTRIBUTE_CUSTOM_REGEX, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_REGEXP_ERROR_DESCRIPTION"));
        }
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_REGEXP);
        fieldNames.add(ATTRIBUTE_CUSTOM_REGEX);
        return fieldNames;
    }

    @Override
    public boolean isCacheable(FormQuestion question)
    {
        AutofillSource source = _getAutoFillSource(question);
        return source == null || source.isCacheable(question);
    }
    
    private Map<AutocompleteOption, List<InputType>> _getAutocompleteOptions()
    {
        Map<AutocompleteOption, List<InputType>> autocompleteOptions = new HashMap<>();
        autocompleteOptions.put(
                new AutocompleteOption("name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_FULLNAME")),
                List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("honorific-prefix", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_HONORIFIC_PREFIX")),
            List.of(InputType.SELECT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("given-name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_GIVEN_NAME")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("family-name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_FAMILY_NAME")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("email", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_EMAIL")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("organization-title", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_ORGANIZATION_TITLE")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("organization", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_ORGANIZATION")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("street-address", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_STREET_ADDRESS")),
            List.of(InputType.TEXT, InputType.TEXTAREA)
        );
        autocompleteOptions.put(
            new AutocompleteOption("country-name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_COUNTRY_NAME")),
            List.of(InputType.TEXT, InputType.SELECT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("postal-code", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_POSTAL_CODE")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("bday", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_BIRTHDAY")),
            List.of(InputType.DATE)
        );
        autocompleteOptions.put(
            new AutocompleteOption("sex", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_SEX")),
            List.of(InputType.SELECT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("tel", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_TEL")),
            List.of(InputType.TEXT)
        );
        autocompleteOptions.put(
            new AutocompleteOption("url", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_URL")),
            List.of(InputType.TEXT)
        );
        
        return autocompleteOptions;
    }
    
    public ModelItem getAutocompleteModelItem(InputType inputType)
    {
        ElementDefinition<String> autocomplete = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_AUTOCOMPLETE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_LABEL", "PLUGINS_FORMS_QUESTIONS_DIALOG_AUTOCOMPLETE_DESC", null);
        StaticEnumerator<String> autocompleteStaticEnumerator = new StaticEnumerator<>();
        
        _getAutocompleteOptions().entrySet()
            .stream()
            .filter(e -> e.getValue().contains(inputType))
            .forEach(e -> autocompleteStaticEnumerator.add(e.getKey().label(), e.getKey().value()));
        
        autocomplete.setEnumerator(autocompleteStaticEnumerator);
        
        return autocomplete;
    }
    
    public ViewElement getAutocompleteViewElement(Model model)
    {
        ViewElement description = new ViewElement();
        description.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_AUTOCOMPLETE));
        return description;
    }
    
    public String getAutocomplete(FormQuestion question)
    {
        return question.getValue(AutocompleteAwareQuestionType.ATTRIBUTE_AUTOCOMPLETE);
    }
    
    @Override
    public Map<String, Object> getAnonymizedData(FormQuestion question)
    {
        return Collections.singletonMap(question.getNameForForm(), isMandatory(question) ? "anonymized" : null);
    }
}
