/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.io.IOException;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.content.FormManager;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.lock.LockHelper;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Post function handling form data after creating and editing it.
 */
public class FormEditionFunction extends AbstractContentWorkflowComponent implements EnhancedFunction
{
    /** Form properties manager. */
    protected FormManager _formManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _formManager = (FormManager) manager.lookup(FormManager.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        _logger.info("Performing form processing workflow function");
        
        // Retrieve current content
        WorkflowAwareContent content = getContent(transientVars);
        UserIdentity user = getUser(transientVars);

        try
        {
            LockableAmetysObject lockableContent = (LockableAmetysObject) content;
            if (lockableContent.isLocked() && !LockHelper.isLockOwner(lockableContent, user))
            {
                throw new WorkflowException("User '" + user + "' try to save content '" + content.getName() + "' but it is locked by another user");
            }
            
            _checkLock (content);
            _formManager.processContentForms(content);
        }
        catch (RepositoryException e)
        {
            throw new WorkflowException("Unable to check lock for content " + content.getName() + "", e);
        }
        catch (AmetysRepositoryException e)
        {
            throw new WorkflowException("Unable to edit content " + content.getName() + " from the repository", e);
        }
        catch (SAXException e)
        {
            _logger.error("An exception occured while parsing the form", e);
        }
        catch (IOException e)
        {
            _logger.error("An exception occured while parsing the form", e);
        }
    }
    
    private void _checkLock(Content content) throws RepositoryException
    {
        if (content instanceof JCRAmetysObject)
        {
            Node node = ((JCRAmetysObject) content).getNode();
         
            if (node.isLocked())
            {
                LockManager lockManager = node.getSession().getWorkspace().getLockManager();
                Lock lock = lockManager.getLock(node.getPath());
                Node lockHolder = lock.getNode();
                
                lockManager.addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
            }
        }
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_EDITION_FUNCTION_LABEL");
    }
}
