/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.plugins.workflow.component.WorkflowArgument;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Worklow condition for checking if current user is the same user as an user defined in a question 
 */
public class IsUserCondition extends AbstractLogEnabled implements EnhancedCondition, Serviceable
{
    /** The question name argument */
    public static final String QUESTION_NAME = "question-name";
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }

    @Override
    public boolean passesCondition(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        FormEntry formEntry = (FormEntry) transientVars.get(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY);
        String questionName = (String) args.get(QUESTION_NAME);
        if (StringUtils.isBlank(questionName) || !formEntry.hasDefinition(questionName) || !formEntry.getType(questionName).getId().equals(ModelItemTypeConstants.USER_ELEMENT_TYPE_ID))
        {
            getLogger().warn("Given argument '" + questionName + "' for condition is not an user form question");
            return false;
        }
        
        UserIdentity userId = (UserIdentity) formEntry.getValue(questionName);
        return userId != null && userId.equals(_currentUserProvider.getUser());
    }
    
    @Override
    public List<WorkflowArgument> getArguments()
    {
        WorkflowArgument questionName = WorkflowElementDefinitionHelper.getElementDefinition(
            QUESTION_NAME,
            new I18nizableText("plugin.forms", "PLUGINS_FORMS_IS_USER_ARGUMENT_QUESTION_NAME_LABEL"),
            new I18nizableText("plugin.forms", "PLUGINS_FORMS_IS_USER_ARGUMENT_QUESTION_NAME_DESCRIPTION"),
            true,
            false
        );
        
        return List.of(questionName);
    }

    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_IS_USER_CONDITION_LABEL");
    }
    
    @Override
    public I18nizableText getFullLabel(Map<String, String> argumentsValues)
    {
        String questionName = argumentsValues.get(QUESTION_NAME);
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_IS_USER_CONDITION_DESCRIPTION", List.of(questionName));
    }
    
    @Override
    public List<WorkflowVisibility> getVisibilities()
    {
        List<WorkflowVisibility> visibilities = EnhancedCondition.super.getVisibilities();
        visibilities.add(WorkflowVisibility.USER);
        return visibilities;
    }
}
