/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.googlecalendar;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.URIUtils;

/**
 * Google calendar iframe generator
 * 
 */
public class CalendarIframeGenerator extends AbstractGenerator
{
    private static final String __CALENDAR_SERVER_URL = "https://calendar.google.com/calendar/embed";

    private static final String __CTZ = "Europe/Paris";

    private static final String __DEFAULT_COLOR = "#2952A3";
    private static final String __DEFAULT_BGCOLOR = "#FFFFFF";
    private static final String __DEFAULT_HEIGHT = "600";
    private static final String __DEFAULT_MODE = "MONTH";
    private static final String __DOES_NOT_SHOW_PARAM = "0";
    
    private static final String __HEIGHT_PARAM = "height";

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("src", _getSrcAttrValue());
        atts.addCDATAAttribute(__HEIGHT_PARAM, _getParameterValue(__HEIGHT_PARAM, __DEFAULT_HEIGHT));
        XMLUtils.startElement(contentHandler, "iframe", atts);
        XMLUtils.endElement(contentHandler, "iframe");
        
        contentHandler.endDocument();
    }
    
    private String _getSrcAttrValue()
    {
        return _buildURL(_getParametersAsMap());
    }
    
    private String _buildURL(Map<String, String> urlParams)
    {
        StringBuilder builder = new StringBuilder(__CALENDAR_SERVER_URL);
        if (urlParams.size() > 0)
        {
            builder.append('?');
        }
        
        Iterator<Entry<String, String>> entrySet = urlParams.entrySet().iterator();
        while (entrySet.hasNext())
        {
            Entry<String, String> entry = entrySet.next();
            builder.append(entry.getKey()).append('=').append(URIUtils.encodeParameter(entry.getValue()));
            if (entrySet.hasNext())
            {
                builder.append('&');
            }
        }
        
        return builder.toString();
    }

    private Map<String, String> _getParametersAsMap()
    {
        Map<String, String> urlParams = new HashMap<>();
        for (String paramName : parameters.getNames())
        {
            if (paramName.startsWith("show"))
            {
                boolean paramValue = parameters.getParameterAsBoolean(paramName, false);
                if (!paramValue)
                {
                    urlParams.put(paramName, __DOES_NOT_SHOW_PARAM);
                }
            }
            else if ("mode".equals(paramName))
            {
                String paramValue = _getParameterValue(paramName, __DEFAULT_MODE);
                if (!__DEFAULT_MODE.equals(paramValue))
                {
                    urlParams.put(paramName, paramValue);
                }
            }
            else if ("bgcolor".equals(paramName))
            {
                String paramValue = _getParameterValue(paramName, __DEFAULT_BGCOLOR);
                urlParams.put(paramName, paramValue);
            }
            else if ("wkst".equals(paramName) || "src".equals(paramName))
            {
                urlParams.put(paramName, parameters.getParameter(paramName, ""));
            }
        }
        urlParams.put("ctz", __CTZ);
        urlParams.put("color", __DEFAULT_COLOR);
        
        return urlParams;
    }
    
    private String _getParameterValue (String name, String defaultValue)
    {
        String value = parameters.getParameter(name, defaultValue);
        if (value.isEmpty())
        {
            value = defaultValue;
        }
        return value;
    }
}
