/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Date;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobExecutionContext;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentVersionHistoryHelper;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.runtime.config.Config;

/**
 * Schedulable to remove all the applications older than a given delay
 */
public class CleanExpiredApplicationSchedulable extends AbstractStaticSchedulable
{
    /** The content DAO */
    protected ContentDAO _contentDAO;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The version history helper */
    protected ContentVersionHistoryHelper _versionHistoryHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _versionHistoryHelper = (ContentVersionHistoryHelper) manager.lookup(ContentVersionHistoryHelper.ROLE);
    }
    
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        Long delay = (Long) context.getMergedJobDataMap().get(Scheduler.PARAM_VALUES_PREFIX + "delay");
        if (delay == null)
        {
            delay = Config.getInstance().getValue("job-offer.clean-expired-application.delay", false, null);
        }
        if (delay == null || delay < 0)
        {
            throw new IllegalArgumentException("The expiration delay must be positive");
        }
        
        ZonedDateTime expirationDateTime = ZonedDateTime.now().minus(Period.ofMonths(delay.intValue()));
        Date expirationDate = DateUtils.asDate(expirationDateTime);
        Expression dateExpression = new DateExpression(DefaultContent.METADATA_MODIFIED, Operator.LE, expirationDate, ExpressionContext.newInstance().withInternal(true));
        
        Expression contentTypeExpression = _cTypeEP.createHierarchicalCTExpression(JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE);
        AndExpression expiredApplicationExpression = new AndExpression(contentTypeExpression, dateExpression);
        String query = ContentQueryHelper.getContentXPathQuery(expiredApplicationExpression);
        try (AmetysObjectIterable<Content> applications = _resolver.query(query))
        {
            _contentDAO.forceDeleteContentsObj(applications, null);
            ForensicLogger.info("data.policy.gdpr.remove.job-offer.applications", Map.of("handled", Long.toString(applications.getSize())), UserPopulationDAO.SYSTEM_USER_IDENTITY);
        }
        
        _versionHistoryHelper.clearUnusedHistory(JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE);
    }
}
