/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer.action;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.captcha.CaptchaHelper;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.joboffer.JobOfferConstants;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.content.FOContentCreationHelper;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

/**
 * Action to apply to a job offer
 */
public class ApplyJobOfferAction extends ServiceableAction
{  
    private static final int _INITIAL_ACTION_ID = 1;
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The JSON Utils */
    protected JSONUtils _jsonUtils;
    
    /** Helper for FO content creation */
    protected FOContentCreationHelper _foContentCreationHelper;
    
    /** The page helper */
    protected PageHelper _pageHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
        _foContentCreationHelper = (FOContentCreationHelper) serviceManager.lookup(FOContentCreationHelper.ROLE);
        _pageHelper = (PageHelper) serviceManager.lookup(PageHelper.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        Multimap<String, I18nizableText> errors = ArrayListMultimap.create();

        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String zoneId = request.getParameter("zoneId");
        ZoneItem zoneItem = _resolver.resolveById(zoneId);
        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        
        String siteName = request.getParameter("site");
        String language = request.getParameter("lang");
        
        ContentType contentType = _cTypeEP.getExtension(JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE);
        String workflowName = contentType.getDefaultWorkflowName().get();
        
        Map<String, Object> values = _foContentCreationHelper.getAndValidateFormValues(request, contentType, "main", errors);
        
        // Check captcha if needed
        if (_pageHelper.isCaptchaRequired(sitemapElement))
        {   
            String captchaValue = request.getParameter("captcha");
            String captchaKey = request.getParameter("captcha-key");

            // Validate the input
            if (!CaptchaHelper.checkAndInvalidate(captchaKey, captchaValue))
            {
                errors.put("captcha", new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_APPLY_FORM_ERROR_INVALID_CAPTCHA"));
            }
        }
        
        // If there are no errors
        if (errors.isEmpty())
        {            
            String title = getTitle(request);
            
            result = _foContentCreationHelper.createAndEditContent(_INITIAL_ACTION_ID, new String[] {JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE}, ArrayUtils.EMPTY_STRING_ARRAY, siteName, title, title, language, values, workflowName, null, new HashMap<>());
            
            if (result.containsKey(Content.class.getName()))
            {
                result.put("success", true);
            }
            else
            {
                result.put("success", false);
            }
        }
        else
        {
            result.put("success", false);
            result.put("errors", _jsonUtils.convertObjectToJson(errors.asMap()));
        }
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }
    
    /**
     * Get title from request
     * @param request the request
     * @return the computed title
     */
    protected String getTitle(Request request)
    {
        Object[] array = List.of(request.getParameter("lastname"), request.getParameter("firstname"))
                .stream()
                .filter(StringUtils::isNotEmpty)
                .collect(Collectors.toList())
                .toArray();
        
        return StringUtils.join(array, " ");
    }

}
