/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer.generator;

import java.io.IOException;
import java.time.LocalDate;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.plugins.joboffer.JobOfferConstants;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.web.WebConstants;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.content.FOContentCreationHelper;
import org.ametys.web.repository.page.Page;

/**
 * Generator for application's form
 */
public class JobApplicationFormGenerator extends ServiceableGenerator
{
    /** The {@link ContentType} manager */
    protected ContentTypeExtensionPoint _cTypeExtPt;
    /** The FO content creation helper */
    protected FOContentCreationHelper _foContentCreationHelper;
    /** The page helper */
    protected PageHelper _pageHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cTypeExtPt = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _foContentCreationHelper = (FOContentCreationHelper) smanager.lookup(FOContentCreationHelper.ROLE);
        _pageHelper = (PageHelper) smanager.lookup(PageHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        Page page = (Page) request.getAttribute(WebConstants.REQUEST_ATTR_PAGE);
        Content content = (Content) request.getAttribute(Content.class.getName());
        
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "form");
        
        LocalDate closingDate = content.getValue(JobOfferConstants.JOB_OFFER_ATTRIBUTE_PATH_CLOSING_DATE);
        if (closingDate != null && closingDate.isBefore(LocalDate.now()))
        {
            XMLUtils.createElement(contentHandler, "expired", "true");
        }
        
        // SAX view
        ContentType contentType = _cTypeExtPt.getExtension(JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE);
        contentType.getView("main").toSAX(contentHandler, DefinitionContext.newInstance()
                                                                           .withItemTagName("metadata"));
        
        // SAX contents values for metadata of type CONTENT
        _foContentCreationHelper.saxContentValues(contentHandler, contentType, "items", content.getLanguage());
        
        if (page != null)
        {
            XMLUtils.createElement(contentHandler, "has-captcha", String.valueOf(_pageHelper.isCaptchaRequired(page)));
        }
        
        XMLUtils.endElement(contentHandler, "form");
        
        contentHandler.endDocument();
    }
    
    

}
