/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer.observer;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.joboffer.JobOfferConstants;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;


/**
 * Abstract observer for job offer
 *
 */
public abstract class AbstractJobOfferObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{
    /** The key for persons in charge in transvient vars */
    protected static final String REQUEST_ATTR_PERSON_IN_CHARGE = AbstractJobOfferObserver.class.getName() + "$personIncharge";
    
    /** The content type helper */
    protected ContentTypesHelper _cTypeHelper;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    private Context _context;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _cTypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    public boolean supports(Event event)
    {
        return _getSupportedEventIds().contains(event.getId()) && getJobOffer(event) != null;
    }
    
    /**
     * Get the current request
     * @return the request
     */
    protected Request _getRequest()
    {
        return ContextHelper.getRequest(_context);
    }
    /**
     * Get the id of supported events
     * @return the id in a Set
     */
    protected abstract Set<String> _getSupportedEventIds();
    
    public int getPriority()
    {
        return MIN_PRIORITY;
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = getJobOffer(event);
        if (content != null)
        {
            _internalObserver(event, content, transientVars);
        }
    }
    
    /**
     * Internal observer
     * @param event the event
     * @param content the job offer content. Cannot be null.
     * @param transientVars The transient variables
     * @throws Exception if an error occurred
     */
    protected abstract void _internalObserver(Event event, Content content, Map<String, Object> transientVars) throws Exception;
    
    /**
     * Get the job offer concerned by the event
     * @param event the event
     * @return the job offer content or null
     */
    protected Content getJobOffer(Event event)
    {
        Map<String, Object> args = event.getArguments();
        
        if (args.containsKey(ObservationConstants.ARGS_CONTENT_ID))
        {
            String contentId = (String) args.get(ObservationConstants.ARGS_CONTENT_ID);
            
            Content content = _resolver.resolveById(contentId);
            return _cTypeHelper.isInstanceOf(content, JobOfferConstants.JOB_OFFER_CONTENT_TYPE) ? content : null;
        }
        
        return null;
    }
}
