/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer.observer;

import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.indexing.IndexingObserver;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.joboffer.JobOfferConstants;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * {@link Observer}  when the job offer has been modified
 */
public class ModifiedJobOfferObserver extends AbstractJobOfferObserver implements IndexingObserver
{
    private SolrIndexer _solrIndexer;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _solrIndexer = (SolrIndexer) smanager.lookup(SolrIndexer.ROLE);
    }
    
    @Override
    protected Set<String> _getSupportedEventIds()
    {
        return Set.of(ObservationConstants.EVENT_CONTENT_MODIFIED);
    }

    @Override
    protected void _internalObserver(Event event, Content content, Map<String, Object> transientVars) throws Exception
    {
        UserIdentity[] personsInCharge = content.getValue(JobOfferConstants.JOB_OFFER_ATTRIBUTE_PATH_PERSON_IN_CHARGE);
        
        UserIdentity[] oldPersonsInCharge = (UserIdentity[]) _getRequest().getAttribute(REQUEST_ATTR_PERSON_IN_CHARGE);
        
        if (!Objects.deepEquals(personsInCharge, oldPersonsInCharge))
        {
            // responsible has changed, update ACL for applications
            _reIndexApplications(content);
        }
    }
    
    private void _reIndexApplications(Content content) throws Exception
    {
        Expression cTypeExpr = new ContentTypeExpression(Operator.EQ, JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE);
        Expression jobExpr = new StringExpression(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_JOB_OFFER, Operator.EQ, content.getId());
        
        Expression finalExpr = new AndExpression(cTypeExpr, jobExpr);
        
        String query = ContentQueryHelper.getContentXPathQuery(finalExpr);
        
        AmetysObjectIterable<Content> applications = _resolver.query(query);
        _solrIndexer.updateAclCache(applications);
    }
}
