/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer.right;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.joboffer.JobOfferConstants;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;

/**
 * {@link AccessController} so responsible of a job offer can access and handle the applications
 *
 */
public class ApplicationAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    /** ContentTypes Helper */
    protected ContentTypesHelper _cTypeHelper;
    /** the ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _cTypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the rights for person in charge of a application content
     * @return the list of allowed rights
     */
    protected List<String> getApplicationRights()
    {
        return List.of(
                "Workflow_Right_Application_Edit",
                "Workflow_Right_Application_Shortlist",
                "Workflow_Right_Application_Disapprove");
    }
    
    /**
     * Determines if the current user is in charge of the current application
     * @param user the user
     * @param content the application content
     * @return true if the current user is in charge
     */
    protected boolean isInCharge(UserIdentity user, Content content)
    {
        UserIdentity[] personsInCharge = getPersonInCharge(content);
        return personsInCharge != null && ArrayUtils.contains(personsInCharge, user);
    }
    
    /**
     * Get the persons in charge of a application
     * @param content the application content
     * @return the persons in charge or null if not found or empty
     */
    protected UserIdentity[] getPersonInCharge(Content content)
    {
        if (content.hasDefinition(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_PERSON_IN_CHARGE))
        {
            return content.getValue(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_PERSON_IN_CHARGE);
        }
        
        return null;
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Content && _cTypeHelper.isInstanceOf((Content) object, JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE);
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (object instanceof Content && isInCharge(user, (Content) object))
        {
            return getApplicationRights().contains(rightId) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        if (object instanceof Content && isInCharge(user, (Content) object))
        {
            return AccessResult.USER_ALLOWED;
        }
        
        return AccessResult.UNKNOWN;
    }

    /**
     * If creator, access to a list of rights
     */
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (isInCharge(user, (Content) object))
        {
            for (String rightId : getApplicationRights())
            {
                permissionByRight.put(rightId, AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByRight;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    /**
     * If right requested is in the list, the creator is added the list of USER_ALLOWED
     */
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (getApplicationRights().contains(rightId))
        {
            UserIdentity[] personInCharge = getPersonInCharge((Content) object);
            if (personInCharge != null)
            {
                for (UserIdentity userIdentity : personInCharge)
                {
                    permissionByUser.put(userIdentity, AccessResult.USER_ALLOWED);
                }
            }
        }
            
        return permissionByUser;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        Map<UserIdentity, AccessResult> readPermissionByUser = new HashMap<>();
        
        UserIdentity[] personInCharge = getPersonInCharge((Content) object);
        if (personInCharge != null)
        {
            for (UserIdentity userIdentity : personInCharge)
            {
                readPermissionByUser.put(userIdentity, AccessResult.USER_ALLOWED);
            }
        }
            
        return readPermissionByUser;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    protected AccessExplanation _getAccessExplanation(AccessResult result, Object object, UserIdentity user, Set<GroupIdentity> groups, String rightId)
    {
        switch (result)
        {
            case USER_ALLOWED:
            case UNKNOWN:
                Content jobApplication = (Content) object;
                ContentValue jobOffer = jobApplication.getValue(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_JOB_OFFER);
                return new AccessExplanation(
                        getId(),
                        result,
                        new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_APPLICATION_ACCESS_CONTROLLER_" + result.name() + "_EXPLANATION",
                                Map.of(
                                        "title", new I18nizableText(jobOffer.getContent().getTitle())
                                        )
                                )
                        );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), result);
        }
    }
    
    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        
        if (StringUtils.isNotBlank(siteName))
        {
            Expression typeExpression = new ContentTypeExpression(Operator.EQ, JobOfferConstants.JOB_OFFER_CONTENT_TYPE);
            Expression inChargeExpression = new UserExpression(JobOfferConstants.JOB_OFFER_ATTRIBUTE_PATH_PERSON_IN_CHARGE, Operator.EQ, identity, true);
            Expression siteExpression = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, siteName, ExpressionContext.newInstance().withInternal(true));
            String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(typeExpression, inChargeExpression, siteExpression));
            
            try (AmetysObjectIterable<Content> offers = _resolver.query(query))
            {
                if (offers.getSize() > 0)
                {
                    List<Expression> applicationsExpression = offers.stream()
                            .map(Content::getId)
                            .<Expression>map(id -> new StringExpression(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_JOB_OFFER, Operator.EQ, id))
                            .toList();
                    
                    String applicationQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(
                            new ContentTypeExpression(Operator.EQ, JobOfferConstants.JOB_APPLICATION_CONTENT_TYPE),
                            new OrExpression(applicationsExpression)
                            ));
                    
                    return _resolver.query(applicationQuery);
                }
            }
        }
        return List.of();
    }
    
    @Override
    protected Collection<String> getHandledRights()
    {
        return getApplicationRights();
    }

    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_APPLICATION_ACCESS_CONTROLLER_CONTEXT_CATEGORY");
    }

    @Override
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Content application)
        {
            ContentValue jobOffer = application.getValue(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_JOB_OFFER);
            return new I18nizableText(jobOffer.getContent().getTitle() + " > " + application.getTitle());
        }
        throw new RightsException("Unsupported object: " + object.toString());
    }
}
