/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.joboffer.workflow;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.joboffer.JobOfferConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;
import org.ametys.web.workflow.SendMailFunction;

import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * OS workflow function to send mail to person(s) in charge when a new application was submitted.
 */
public class SendMailToPersonInChargeFunction extends SendMailFunction
{
    private RenderingContextHandler _renderingContextHandler;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _renderingContextHandler = (RenderingContextHandler) smanager.lookup(RenderingContextHandler.ROLE);
    }
    
    
    @Override
    protected Set<UserIdentity> _getUsers(WorkflowAwareContent content, Set<String> rights) throws WorkflowException
    {
        if (content.hasDefinition(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_PERSON_IN_CHARGE))
        {
            UserIdentity[] personIncharge = content.getValue(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_PERSON_IN_CHARGE);
            return personIncharge != null ? Set.of(personIncharge) : Set.of();
        }
        
        return Set.of();
    }
    
   
    
    @Override
    protected String getSender(Map transientVars, WorkflowAwareContent content) throws WorkflowException
    {
        if (content instanceof WebContent)
        {
            Site site = ((WebContent) content).getSite();
            return site.getValue("site-mail-from", false, Config.getInstance().getValue("smtp.mail.from"));
        }
        else
        {
            return Config.getInstance().getValue("smtp.mail.from");
        }
    }
    
    @Override
    protected List<String> getSubjectI18nParams(User user, WorkflowAwareContent content)
    {
        ContentValue value = content.getValue(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_JOB_OFFER);
        Content jobOffer = value.getContent();
        
        List<String> i18nParams = new ArrayList<>();
        i18nParams.add(jobOffer.getTitle()); // {0}
        i18nParams.add(jobOffer.getValue(JobOfferConstants.JOB_OFFER_ATTRIBUTE_PATH_REF_ID, false, "")); // {1}
        
        return i18nParams;
    }
    
    @Override
    protected MailBodyBuilder getMailBody(String subjectI18nKey, String bodyI18nKey, User user, WorkflowAwareContent content, Map transientVars)
    {
        Source src = null;

        Request request = _getRequest();
        String currentWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        
        try
        {
            // Force default workspace and BACK rendering context to resolve uri (application contents are never published in live)
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, "default");
            _renderingContextHandler.setRenderingContext(RenderingContext.PREVIEW);
            
            request.setAttribute("_baseServerPath", _getRequestUri());
            
            ContentValue jobOffer = content.getValue(JobOfferConstants.JOB_APPLICATION_ATTRIBUTE_PATH_JOB_OFFER);
            Content jobOfferContent = jobOffer.getContent();
            
            Site site = _getSite(content);
            String jobApplicationUri = _getContentUri(content, null, site);
            
            List<String> i18nParams = List.of(jobOfferContent.getTitle(), jobOfferContent.getValue(JobOfferConstants.JOB_OFFER_ATTRIBUTE_PATH_REF_ID, false, StringUtils.EMPTY));
            
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(getMailSubject(subjectI18nKey, user, content, transientVars))
                .addMessage(new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_WORKFLOW_MAIL_BODY_INTRO", i18nParams))
                .addMessage(new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_WORKFLOW_MAIL_BODY_JOB_APPLICATION_LINK", List.of(jobApplicationUri)))
                .addMessage(new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_WORKFLOW_MAIL_BODY_TOOL_LINK", List.of(_getToolUri(site))))
                .withLink(jobApplicationUri, new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_WORKFLOW_MAIL_BODY_JOB_APPLICATION_LINK_TITLE"));
            
            String uri = "cocoon:/apply/mail.html";
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("content", content);
            parameters.put("uriPrefix", _getRequestUri());
            parameters.put("siteName", _getSite(content).getName());

            src = _sourceResolver.resolveURI(uri, null, parameters);
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            String details = IOUtils.toString(reader);
            
            bodyBuilder.withDetails(new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_WORKFLOW_MAIL_BODY_JOB_APPLICATION"), details, false);
            
            return bodyBuilder;
        }
        catch (IOException e)
        {
            _logger.warn("Failed to get HTML body for mail notification for new application " + content.getId(), e);
            return null;
        }
        finally
        {
            _sourceResolver.release(src);
            
            _renderingContextHandler.setRenderingContext(currentContext);
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWorkspace);
        }
    }
    
    /**
     * Get the tool uri
     * @param site the current site
     * @return the tool uri
     */
    protected String _getToolUri(Site site)
    {
        String siteName;
        if (site != null)
        {
            siteName = site.getName();
        }
        else
        {
            // fallback to the current site if no site is provided
            siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        }
        
        return _getRequestUri() + "/" + siteName + "/index.html?uitool=uitool-application-search,id:%27search-ui.job-applications%27";
    }
    
    @Override
    protected void _sendMails(I18nizableText subjectI18nKey, MailBodyBuilder bodyBuilder, Map<String, Set<String>> recipientsByLanguage, String from) throws IOException
    {
        if (bodyBuilder != null)
        {
            for (String language : recipientsByLanguage.keySet())
            {
                String subject = _i18nUtils.translate(subjectI18nKey, language);
                String body = bodyBuilder.withLanguage(language).build();
                
                for (String recipient : recipientsByLanguage.get(language))
                {
                    try
                    {
                        SendMailHelper.newMail()
                                      .withSubject(subject)
                                      .withHTMLBody(body)
                                      .withSender(from)
                                      .withRecipient(recipient)
                                      .withAsync(true)
                                      .sendMail();
                    }
                    catch (MessagingException | IOException e)
                    {
                        _logger.warn("Could not send a workflow notification mail to " + recipient, e);
                    }
                }
            }
        }
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.job-offer", "PLUGINS_JOB_OFFER_SEND_MAIL_TO_PERSON_IN_CHARGE_FUNCTION_LABEL");
    }
    
}
