/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

function migrateThemes(session)
{
    logger.info("Migrate themes");
    var count = 0;
    const nodes = Repository.query("//element(*, ametys:directoryTheme)", false, session);
    while (nodes.hasNext())
    {
        count++;
        const node = nodes.next();
        migrateTheme("directoryTheme://" + node.getIdentifier(), node, session);
    }
    logger.info(count + " theme(s) migrated");
    logger.info("====================");
}

function migrateLinks(session)
{
    logger.info("Migrate links");
    var count = 0;
    const nodes = Repository.query("//element(*, ametys:directoryLink)[@ametys-internal:themes]", false, session);
    while (nodes.hasNext())
    {
        count++;
        const node = nodes.next();
        migrateTagsInANode(node, "ametys-internal:themes", session);
    }
    logger.info(count + " links(s) migrated");
    logger.info("====================");
}

function migrateTagsInANode(node, themePropertyName, session)
{
    var tags = node.getProperty(themePropertyName).getValues();
    logger.debug("Node '" + node.getIdentifier() + "' have " + tags.length + " themes");

    if (tags.length > 0)
    {
        var newTags = [];
        for (var i = 0; i < tags.length; i++)
        {
            var newTagName = migrateTheme(tags[i].getString(), node, session);
            if (newTagName != null)
            {
                newTags.push(newTagName);
            }
        }

        Repository.helper.setProperty(node, themePropertyName, newTags);

        node.save();
    }
}

function migrateTheme(themeId, parentNode, session)
{
    logger.debug("migrateTheme with id='" + themeId + "'");
    if (!themeId.startsWith("directoryTheme://"))
    {
        logger.warn("There is already a new theme used in this repository : '" + themeId + "', it will not be migrated.");
        return themeId;
    }
    
    try
    {
        // Get theme name
        var oldTheme = Repository.resolver.resolveById(themeId, session);
        var name = oldTheme.getName();
        var label = oldTheme.getLabel();
        var siteName = oldTheme.getSiteName();
        var language = oldTheme.getLanguage();
    
        var languageNode = oldTheme.getParent().getParent();
        var themesNode;
        
        if (languageNode.hasChild("themes"))
        {
            themesNode = languageNode.getChild("themes");
        }
        else
        {
            themesNode = languageNode.createChild("themes", "ametys:themes");
            languageNode.saveChanges();
        }
        
        // If it does not exist, create a new link
        if (themesNode.hasChild(name))
        {
            return name;
        }
        else
        {
            logger.debug("Migrate theme '" + label + "' in site '" + siteName + "' in language '" + language + "'");
            
            newTheme = themesNode.createChild(name, "ametys:theme");
            newTheme.setTitle(label);
            
            themesNode.saveChanges();
            newTheme.saveChanges();
    
            logger.debug("new theme created with id='" + newTheme.getId() + "'");
            return newTheme.getName();
        }
    }
    catch (e)
    {
        var parentId = parentNode ? parentNode.getIdentifier() : "";
        logger.warn("Impossible to find the theme with id '" + themeId + "', parent id : '" + parentId + "'");
    }
    
    return null;
}

function migrateServices(session)
{
    logger.info("Migrate services");
    var count = 0;
    const nodes = Repository.query("//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.plugins.linkdirectory.DirectoryService']/ametys:service_parameters", false, session);
    while (nodes.hasNext())
    {
        count++;
        const node = nodes.next();
        
        migrateTagsInANode(node, "ametys:themes", session);
    }
    logger.info(count + " service(s) migrated");
    logger.info("====================");
}

function removeOldThemes(session)
{
    logger.info("Remove old themes");
    var count = 0;
    const nodes = Repository.query("//element(*, ametys:directoryThemes)", false, session);
    while (nodes.hasNext())
    {
        count++;
        const node = nodes.next();
        var parent = node.getParent();
        node.remove();
        
        parent.save();
    }
    logger.info(count + " old theme(s) removed");
    logger.info("====================");
}

function migrateAll(session)
{
    migrateThemes(session);
    
    migrateLinks(session);
    
    migrateServices(session);
    
    removeOldThemes(session);
}

migrateAll(Repository.session);
migrateAll(Repository.liveSession);
