/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.io.InputStream;

import org.ametys.cms.data.Binary;
import org.ametys.cms.indexing.solr.SolrAclCacheUninfluentialObject;
import org.ametys.plugins.repository.AmetysRepositoryException;

/**
 * A link in the directory.
 */
@SolrAclCacheUninfluentialObject
public interface Link
{
    /** Type of a link. */
    public enum LinkType 
    {
        /** External URL */
        URL,
        /** Internal page */
        PAGE
    }
    
    /** Status of a link. */
    public enum LinkStatus 
    {
        /** Normal link */
        NORMAL,
        /** New link */
        NEW,
        /** Broken link */
        BROKEN
    }
    
    /** Default visibility of a link. */
    public enum LinkVisibility 
    {
        /** Visible by default */
        VISIBLE,
        /** Hidden by default */
        HIDDEN,
    }

    /**
     * Get the URL.
     * 
     * @return the link URL.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getUrl() throws AmetysRepositoryException;

    /**
     * Set the URL.
     * 
     * @param type The URL type
     * @param url the link URL to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setUrl(LinkType type, String url) throws AmetysRepositoryException;

    /**
     * Get the internal URL.
     * 
     * @return the link internal URL.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getInternalUrl() throws AmetysRepositoryException;

    /**
     * Set the internal URL.
     * 
     * @param url the link internal URL to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setInternalUrl(String url) throws AmetysRepositoryException;

    /**
     * Retrieves the linked URL type.
     * 
     * @return the linked URL type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public LinkType getUrlType() throws AmetysRepositoryException;

    /**
     * Get the link title.
     * 
     * @return the link title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException;

    /**
     * Set the title.
     * 
     * @param title the link title to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException;

    /**
     * Get the link content.
     * 
     * @return the link content.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getContent() throws AmetysRepositoryException;

    /**
     * Set the content.
     * 
     * @param content the link content to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setContent(String content) throws AmetysRepositoryException;

    /**
     * Get the link alternative.
     * 
     * @return the link alternative.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getAlternative() throws AmetysRepositoryException;

    /**
     * Set the link alternative.
     * 
     * @param alternative the link alternative to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setAlternative(String alternative) throws AmetysRepositoryException;

    /**
     * Get the picture as a binary metadata.
     * 
     * @return the picture as a binary metadata.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Binary getExternalPicture() throws AmetysRepositoryException;

    /**
     * Set the picture from an external file.
     * 
     * @param mimeType the file MIME type.
     * @param filename the file name.
     * @param stream an input stream on the file bytes.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setExternalPicture(String mimeType, String filename, InputStream stream) throws AmetysRepositoryException;

    /**
     * Get the picture resource ID.
     * 
     * @return the resource ID.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getResourcePictureId() throws AmetysRepositoryException;

    /**
     * Set the picture from an explorer resource.
     * 
     * @param resourceId the resource ID.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setResourcePicture(String resourceId) throws AmetysRepositoryException;

    /**
     * Removes any picture currently assigned.
     * 
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setNoPicture() throws AmetysRepositoryException;

    /**
     * Get the picture type.
     * 
     * @return the picture type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getPictureType() throws AmetysRepositoryException;

    /**
     * Set the picture type.
     * 
     * @param type the picture type to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setPictureType(String type) throws AmetysRepositoryException;

    /**
     * Get the picture glyph.
     * 
     * @return the picture glyph.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getPictureGlyph() throws AmetysRepositoryException;

    /**
     * Set the picture glyph.
     * 
     * @param glyph the glyph picture to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setPictureGlyph(String glyph) throws AmetysRepositoryException;
    
    /**
     * Get the picture alternative.
     * 
     * @return the picture alternative.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getPictureAlternative() throws AmetysRepositoryException;

    /**
     * Set the picture alternative.
     * 
     * @param alternative the picture alternative to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setPictureAlternative(String alternative) throws AmetysRepositoryException;

    /**
     * Get the themes.
     * 
     * @return the themes in an Array
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String[] getThemes() throws AmetysRepositoryException;

    /**
     * Set the themes.
     * 
     * @param themes the themes to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setThemes(String[] themes) throws AmetysRepositoryException;

    /**
     * Remove theme if exist
     * 
     * @param themeId The id of theme to remove
     * @throws AmetysRepositoryException if an error occurs
     */
    public void removeTheme(String themeId) throws AmetysRepositoryException;

    /**
     * Get the id of the provider of dynamic information.
     * @return the id of provider. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getDynamicInformationProvider() throws AmetysRepositoryException;

    /**
     * Set the id of the provider of dynamic information.
     * @param providerId the id of provider
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setDynamicInformationProvider(String providerId) throws AmetysRepositoryException;
    
    /**
     * Get the color.
     * 
     * @return the color.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getColor() throws AmetysRepositoryException;

    /**
     * Set the color.
     * 
     * @param color the color to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setColor(String color) throws AmetysRepositoryException;
    
    /**
     * Get the page.
     * 
     * @return the page.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getPage() throws AmetysRepositoryException;

    /**
     * Set the page.
     * 
     * @param pageId the page id to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setPage(String pageId) throws AmetysRepositoryException;
    
    /**
     * Get the status.
     * 
     * @return the status.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public LinkStatus getStatus() throws AmetysRepositoryException;

    /**
     * Set the status.
     * @param status the status to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setStatus(LinkStatus status) throws AmetysRepositoryException;
    
    /**
     * Get the default visibility.
     * @return the default visibility.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public LinkVisibility getDefaultVisibility() throws AmetysRepositoryException;

    /**
     * Set the default visibility.
     * @param visibility the default visibility to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setDefaultVisibility(LinkVisibility visibility) throws AmetysRepositoryException;
}
