/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;

/**
 * Create or set user preferences attached to a link directory service or the link directory input data mode for front-end users
 */
public class LinkDirectorySetUserPreferencesAction extends ServiceableAction
{
    /** The user preferences manager. */
    private UserPreferencesManager _userPrefManager;

    /** The current user provider */
    private CurrentUserProvider _currentUserProvider;

    private DirectoryHelper _directoryHelper;
    
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userPrefManager = (UserPreferencesManager) serviceManager.lookup(UserPreferencesManager.ROLE + ".FO");
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _directoryHelper = (DirectoryHelper) serviceManager.lookup(DirectoryHelper.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        // We save user preferences for the FO or BO users
        UserIdentity user = _currentUserProvider.getUser();
        if (user != null && StringUtils.isNotEmpty(user.getLogin()) && StringUtils.isNotEmpty(user.getPopulationId()))
        {
            String zoneItemId = request.getParameter("saving-id"); // can be empty (from some input data)
            
            String storageContext = _directoryHelper.getStorageContext(request, zoneItemId);
            Map<String, String> contextVars = _directoryHelper.getContextVars(request);
            
            _setLinksPositionUserPref(request, storageContext, contextVars, user);
            _setHiddenLinksUserPref(request, storageContext, contextVars, user);
            
            request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        }
        
        return EMPTY_MAP;
    }

    /**
     * Set the links position user pref
     * @param request the request
     * @param storageContext the storage context
     * @param contextVars the context vars
     * @param user the current user
     * @throws UserPreferencesException If failed to save user preferences
     */
    private void _setLinksPositionUserPref(Request request, String storageContext, Map<String, String> contextVars, UserIdentity user) throws UserPreferencesException
    {
        String linksPosition = _getLinksPosition(request);
        
        // If null, we don't change the user pref.
        // However, if empty, we set to empty the user pref. 
        if (linksPosition != null)  
        {
            // TODO it would be nice to change the name of this user pref but the storage is still the same, so for the moment we avoid the SQL migration
            // Cf issue LINKS-141
            // Change in org.ametys.plugins.linkdirectory.DirectoryHelper#saxLinks too
            
            _userPrefManager.addUserPreference(user, storageContext, contextVars, DirectoryHelper.USER_PREF_ORDERED_LINK_ATTR, linksPosition);
        }
    }
    
    /**
     * Get the ordered link ids from the request
     * @param request the request
     * @return the comma-separated string of ordered links ids
     */
    private String _getLinksPosition(Request request)
    {
        return request.getParameter("ordered-links");
    }
    
    /**
     * Set the hidden links user pref
     * @param request the request
     * @param storageContext the storage context
     * @param contextVars the context vars
     * @param user the current user
     * @throws UserPreferencesException if failed to save user preferences
     */
    private void _setHiddenLinksUserPref(Request request, String storageContext, Map<String, String> contextVars, UserIdentity user) throws UserPreferencesException
    {
        String hiddenLinks = _getHiddenLinks(request);
        
        // If null, we don't change the user pref.
        // However, if empty, we set to empty the user pref. 
        if (hiddenLinks != null)  
        {
            _userPrefManager.addUserPreference(user, storageContext, contextVars, DirectoryHelper.USER_PREF_HIDDEN_LINK_ATTR, hiddenLinks);
        }
        
    }
    
    /**
     * Get the hidden link ids from the request
     * @param request the request
     * @return the comma-separated string of hidden links ids
     */
    private String _getHiddenLinks(Request request)
    {
        return request.getParameter("hidden-links");
    }
}
