/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceFactory;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.BinarySource;
import org.ametys.core.util.FilenameUtils;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * {@link SourceFactory} for link data.
 */
public class LinkSourceFactory extends AbstractLogEnabled implements SourceFactory, ThreadSafe, Serviceable
{
    /** source scheme */
    public static final String SCHEME = "link";
    
    private AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public Source getSource(String location, Map parameters) throws IOException, MalformedURLException
    {
        String uri = location.substring((SCHEME + "://").length());
        
        // uri are like <contentPath>|<attributePath>;<filename>
        int i = uri.indexOf('|');
        int j = uri.indexOf(';', i);
        String path = uri.substring(0, i);
        String attribute = uri.substring(i + 1, j);
        String filename = uri.substring(j + 1);
        
        try
        {
            DefaultLink link = _resolver.resolveByPath(FilenameUtils.decode(path));
            
            Binary binary = link.getValue(attribute);
    
            return new BinarySource(binary, FilenameUtils.decode(filename), location, SCHEME);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to resolve binary attribute for uri " + location, e);
            
            // returns an unexisting Source
            return new BinarySource(null, FilenameUtils.decode(filename), location, SCHEME);
        }
    }

    public void release(Source source)
    {
        // nothing to do
    }
}
