/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.dynamic;

import java.io.IOException;
import java.util.Date;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationException.ExceptionType;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Abstract generator for saxing dynamic information.
 */
public abstract class AbstractInternalDynamicInformationGenerator extends ServiceableGenerator
{
    /** Constant for root tag */
    public static final String DYNAMIC_INFORMATION = "dynamic-information";
    /** Constant for short value tag */
    public static final String SHORT_VALUE = "short-value";
    /** Constant for long value tag */
    public static final String LONG_VALUE = "long-value";
    /** Constant for tooltip tag */
    public static final String TOOLTIP = "tooltip";
    /** Constant for item tag */
    public static final String ITEM = "item";
    /** Constant for item's title tag */
    public static final String ITEM_TITLE = "title";
    /** Constant for item's summary tag */
    public static final String ITEM_SUMMARY = "summary";
    /** Constant for item's footer tag */
    public static final String ITEM_FOOTER = "footer";
    /** Constant for item's error message tag */
    public static final String ERROR = "error";
    /** Constant for item's error type tag */
    public static final String ERROR_ATTRIBUTE_TYPE = "type";

    /** The current user provider **/
    protected CurrentUserProvider _currentUserProvider;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();

        XMLUtils.startElement(contentHandler, DYNAMIC_INFORMATION);

        try
        {
            saxShortValue();
            saxLongValue();

            XMLUtils.startElement(contentHandler, TOOLTIP);
            saxTooltips();
            XMLUtils.endElement(contentHandler, TOOLTIP);
        }
        catch (DynamicInformationException e)
        {
            saxError(e);
        }

        XMLUtils.endElement(contentHandler, DYNAMIC_INFORMATION);

        contentHandler.endDocument();
    }
    
    /** 
     * Get the current connected user
     * @return A user identity or <code>null</code> if there is no connected user
     */
    protected UserIdentity getCurrentUser()
    {
        return _currentUserProvider.getUser();
    }
    
    /**
     * Get the current language
     * @return the current language
     */
    protected String getCurrentLanguage()
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        return request.getParameter("lang");
    }

    /**
     * Get the number of requested max items
     * @return the number of max items. -1 if there is no limitations
     */
    protected int getMaxItems() 
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        try
        {
            return Integer.parseInt(request.getParameter("maxItems"));
        }
        catch (NumberFormatException e)
        {
            return -1;
        }
    }
    
    /**
     * SAX the error 
     * @param e The thrown exception
     * @throws SAXException If a SAXException error occurs
     */
    protected void saxError(DynamicInformationException e) throws SAXException
    {
        if (e.isSilent())
        {
            getLogger().warn("Unable to sax dynamic information", e);
        }
        else
        {
            getLogger().error("Unable to sax dynamic information", e);
        }
        
        ExceptionType type = e.getType();
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute(ERROR_ATTRIBUTE_TYPE, type.name());
        XMLUtils.startElement(contentHandler, ERROR, attr);
        I18nizableText errorMessage = new I18nizableText("plugin.link-directory", "PLUGINS_LINKDIRECTORY_DYNAMIC_INFO_PROVIDER_ERROR_" + type.name().toUpperCase());
        errorMessage.toSAX(contentHandler);
        XMLUtils.endElement(contentHandler, ERROR);
    }
    
    /**
     * Create a short value node.
     * @throws SAXException If a SAXException error occurs
     * @throws DynamicInformationException if an error occurs while computing the dynamic information
     */
    protected abstract void saxShortValue() throws SAXException, DynamicInformationException;

    /**
     * Create a long value node.
     * @throws SAXException If a error occurs while saxing
     * @throws DynamicInformationException if an error occurs while computing the dynamic information
     */
    protected abstract void saxLongValue() throws SAXException, DynamicInformationException;

    /**
     * Create a tooltip node.
     * @throws SAXException If a error occurs while saxing
     * @throws DynamicInformationException if an error occurs while computing the dynamic information
     */
    protected abstract void saxTooltips() throws SAXException, DynamicInformationException;

    /**
     * Create an item node.
     * 
     * @param title The title of item.
     * @param summary The summary of item.
     * @param footer The footer of item.
     * @throws SAXException If a error occurs while saxing
     * @throws DynamicInformationException if an error occurs while computing the dynamic information
     */
    protected void saxItem(String title, String summary, String footer) throws SAXException, DynamicInformationException
    {
        XMLUtils.startElement(contentHandler, ITEM);
        XMLUtils.createElement(contentHandler, ITEM_TITLE, title == null ? "" : title);
        XMLUtils.createElement(contentHandler, ITEM_SUMMARY, summary == null ? "" : summary);
        XMLUtils.createElement(contentHandler, ITEM_FOOTER, footer == null ? "" : footer);
        XMLUtils.endElement(contentHandler, ITEM);
    }
    
    /**
     * Create an item node when the title is a {@link Date}
     * @param title The title of item as Date.
     * @param summary The summary of item.
     * @param footer The footer of item.
     * @throws SAXException If a error occurs while saxing
     * @throws DynamicInformationException if an error occurs while computing the dynamic information
     */
    protected void saxItem(Date title, String summary, String footer) throws SAXException, DynamicInformationException
    {
        XMLUtils.startElement(contentHandler, ITEM);
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("type", "date");
        XMLUtils.createElement(contentHandler, ITEM_TITLE, attr, DateUtils.dateToString(title));
        XMLUtils.createElement(contentHandler, ITEM_SUMMARY, summary == null ? "" : summary);
        XMLUtils.createElement(contentHandler, ITEM_FOOTER, footer == null ? "" : footer);
        XMLUtils.endElement(contentHandler, ITEM);
    }
}
