/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.link;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.linkdirectory.Link;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;

/**
 * Removes a user link in Ametys.
 */
public class RemoveUserLinkAction extends ServiceableAction
{
    /** The DAO for {@link Link}s */
    protected LinkDAO _linkDAO;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _linkDAO = (LinkDAO) smanager.lookup(LinkDAO.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, Object> result = new HashMap<>();
        
        UserIdentity user = _currentUserProvider.getUser();
        
        // Check user authenticated
        if (user != null)
        {
            String idsAsString = request.getParameter("ids");
            Object[] array = _jsonUtils.convertJsonToArray(idsAsString);
            
            List<String> linksToRemove = new ArrayList<>();
            for (Object id : array)
            {
                if (id instanceof String)
                {
                    linksToRemove.add((String) id);
                }
            }
            
            _deleteLinks(request, linksToRemove);
            result.put("error", "none");
        }
        else
        {
            result.put("error", "unauthenticated-user");
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    private List<String> _deleteLinks(Request request, List<String> ids)
    {
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            return _linkDAO.deleteLinks(ids);
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
}
