/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.tracking;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.Tag;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.LinkDirectoryThemesInputDataHelper;
import org.ametys.plugins.linkdirectory.LinkDirectoryThemesInputDataHelper.ConfiguredThemesInputData;
import org.ametys.plugins.linkdirectory.LinkDirectoryThemesInputDataHelper.ThemesInputData;
import org.ametys.plugins.linkdirectory.theme.ThemesDAO;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.webanalytics.matomo.tracking.AbstractMatomoEventTrackingProvider;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * The Matomo tracking provider for the link directory
 */
public class LinkDirectoryMatomoTrackingProvider extends AbstractMatomoEventTrackingProvider implements Serviceable
{
    /** The user preferences manager. */
    protected UserPreferencesManager _userPreferencesManager;
    
    /** The link directory themes input data helper */
    protected LinkDirectoryThemesInputDataHelper _linkDirectoryThemesInputDataHelper;
    
    /** Themes DAO */
    protected ThemesDAO _themesDAO;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userPreferencesManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE + ".FO");
        _linkDirectoryThemesInputDataHelper = (LinkDirectoryThemesInputDataHelper) manager.lookup(LinkDirectoryThemesInputDataHelper.ROLE);
        _themesDAO = (ThemesDAO) manager.lookup(ThemesDAO.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    protected List<MatomoEvent> getEvents(Site site) 
    {
        List<MatomoEvent> requests = new ArrayList<>();
        
        AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps();
        for (Sitemap sitemap : sitemaps)
        {
            requests.addAll(_getConfigurableLinkServiceEvents(site, sitemap));
            requests.addAll(_getConfigurableLinkInputDataEvents(site, sitemap));
        }
        
        return requests;
    }
    
    private List<MatomoEvent> _getConfigurableLinkServiceEvents(Site site, Sitemap sitemap)
    {
        List<MatomoEvent> requests = new ArrayList<>();

        String siteName = site.getName();
        String language = sitemap.getSitemapName();
        
        for (ModifiableZoneItem zoneItem : _getLinkDirectoryServiceZoneItems(siteName))
        {
            ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
            String[] themeIds = serviceParameters.getValue("themes");
            for (String themeId : themeIds)
            {
                try
                {
                    String themeLabel = _getThemeLabel(themeId, siteName, language);
                    if (StringUtils.isNotBlank(themeLabel))
                    {
                        requests.add(new MatomoEvent(
                            language + "/" + themeLabel, 
                            _getNbUsersWithCustomLinks(siteName, language, zoneItem.getId())
                        ));
                    }
                }
                catch (UserPreferencesException e)
                {
                    getLogger().error("An error occurred getting user preferences for link directories. No request is sent to Matomo.", e);
                }
            }
            
        }

        return requests;
    }
    
    private AmetysObjectIterable<ModifiableZoneItem> _getLinkDirectoryServiceZoneItems(String siteName)
    {
        String xpathQuery = "//element(" + siteName + ", ametys:site)//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.plugins.linkdirectory.DirectoryService' and ametys:service_parameters/@ametys:configurable = 'true']";
        return _resolver.query(xpathQuery);
    }

    private List<MatomoEvent>  _getConfigurableLinkInputDataEvents(Site site, Sitemap sitemap)
    {
        List<MatomoEvent> requests = new ArrayList<>();

        String siteName = site.getName();
        String language = sitemap.getSitemapName();

        ConfiguredThemesInputData configuredThemes = _linkDirectoryThemesInputDataHelper.getThemesInputData(site.getSkinId());
        String error = configuredThemes.error();
        if (StringUtils.isBlank(error) && configuredThemes.themesInputDatas() != null)
        {
            for (ThemesInputData themesInputData : configuredThemes.themesInputDatas())
            {
                try
                {
                    if (themesInputData.configurable()) // Send only configurable themes
                    {
                        String themeId = themesInputData.themes()
                                .stream()
                                .filter(t -> t.get("lang") == null || t.get("lang").equals(language))
                                .map(t -> t.get("id"))
                                .findFirst()
                                .orElse(null);
                        
                        String themeLabel = _getThemeLabel(themeId, siteName, language);
                        if (StringUtils.isNotBlank(themeLabel))
                        {
                            requests.add(new MatomoEvent(
                                language + "/" + themeLabel, 
                                _getNbUsersWithCustomLinks(siteName, language, themesInputData.id())
                            ));
                        }
                    }
                }
                catch (UserPreferencesException e)
                {
                    getLogger().error("An error occurred getting user preferences for link directories. No request is sent to Matomo.", e);
                }
            }
        }
        
        return requests;
    }
    
    private String _getThemeLabel(String themeId, String siteName, String language)
    {
        Tag theme = StringUtils.isNotBlank(themeId)
                ? _themesDAO.getTag(themeId, Map.of("language", language, "siteName", siteName))  
                : null;
        
        return theme != null ? _i18nUtils.translate(theme.getTitle(), language) : null;
    }
    
    private long _getNbUsersWithCustomLinks(String siteName, String language, String thematicId) throws UserPreferencesException
    {
        String storageContext = siteName + "/" + language + "/" + thematicId;
        Map<String, String> contextVars = Map.of(
            FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName,
            FOUserPreferencesConstants.CONTEXT_VAR_LANGUAGE, language
        );
        
        return _userPreferencesManager.getAllUnTypedUserPrefs(storageContext, contextVars)
                .values()
                .stream()
                .filter(this::_hasCustomLinks)
                .count();
    }
    
    private boolean _hasCustomLinks(Map<String, String> userPrefs)
    {
        String orderedLinksAsString = userPrefs.get(DirectoryHelper.USER_PREF_ORDERED_LINK_ATTR);
        String[] orderedLinks = StringUtils.split(orderedLinksAsString, ",");
        
        String hiddenLinksAsString = userPrefs.get(DirectoryHelper.USER_PREF_HIDDEN_LINK_ATTR);
        String[] hiddenLinks = StringUtils.split(hiddenLinksAsString, ",");
        
        // Links are customized if there is some ordered links or some hidden links in the users preferences
        return orderedLinks != null && orderedLinks.length > 0 || hiddenLinks != null && hiddenLinks.length > 0;
    }
}
