/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import io.github.jav.exposerversdk.ExpoPushError;
import io.github.jav.exposerversdk.ExpoPushMessage;
import io.github.jav.exposerversdk.ExpoPushMessageTicketPair;
import io.github.jav.exposerversdk.ExpoPushTicket;
import io.github.jav.exposerversdk.PushClient;
import io.github.jav.exposerversdk.PushClientCustomData;
import io.github.jav.exposerversdk.PushClientException;
import io.github.jav.exposerversdk.PushNotificationErrorsException;
import io.github.jav.exposerversdk.PushNotificationException;

/**
 * manager to push notifications to users
 */
public class PushNotificationManager extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = PushNotificationManager.class.getName();

    /** User Preferences Helper */
    protected UserPreferencesHelper _userPreferencesHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _userPreferencesHelper = (UserPreferencesHelper) manager.lookup(UserPreferencesHelper.ROLE);
    }

    /**
     * Push a notification to a list of registered devices
     * @param title title of the message
     * @param message message to send
     * @param tokens list of devicet tokens
     * @param data the data to push
     * @throws PushClientException something went wrong
     */
    public void pushNotifications(String title, String message, Map<UserIdentity, Set<String>> tokens, Map<String, Object> data) throws PushClientException
    {
        getLogger().debug("Pushing message '{}' with tokens {}", title, tokens);

        // if something is missing : https://github.com/jav/expo-server-sdk-java
        List<String> validTokens = tokens.values()
                .stream()
                .flatMap(Set::stream)
                .filter(PushClientCustomData::isExponentPushToken)
                .toList();

        if (validTokens.size() > 0)
        {
            ExpoPushMessage expoPushMessage = new ExpoPushMessage();
            expoPushMessage.addAllTo(validTokens);
            expoPushMessage.setTitle(title);
            expoPushMessage.setBody(message);

            expoPushMessage.setData(data);


            List<ExpoPushMessage> expoPushMessages = new ArrayList<>();
            expoPushMessages.add(expoPushMessage);

            PushClient client = new PushClient();
            List<List<ExpoPushMessage>> chunks = client.chunkPushNotifications(expoPushMessages);

            List<CompletableFuture<List<ExpoPushTicket>>> messageRepliesFutures = new ArrayList<>();

            for (List<ExpoPushMessage> chunk : chunks)
            {
                messageRepliesFutures.add(client.sendPushNotificationsAsync(chunk));
            }
            
            // Invert map, so we can easily retrieve the user from the token
            Map<String, UserIdentity> tokenToUserMap = new HashMap<>();
            for (Entry<UserIdentity, Set<String>> entry : tokens.entrySet())
            {
                for (String token : entry.getValue())
                {
                    tokenToUserMap.put(token, entry.getKey());
                }
            }

            // Wait for each completable future to finish
            List<ExpoPushTicket> allTickets = new ArrayList<>();
            List<ExpoPushMessage> expoPushMessagesHandled = new ArrayList<>();
            int count = 0;
            for (CompletableFuture<List<ExpoPushTicket>> messageReplyFuture : messageRepliesFutures)
            {
                try
                {
                    for (ExpoPushTicket ticket : messageReplyFuture.get())
                    {
                        if (count < expoPushMessages.size())
                        {
                            expoPushMessagesHandled.add(expoPushMessages.get(count));
                            allTickets.add(ticket);
                            count++;
                        }
                        else
                        {
                            getLogger().warn("More tickets were received than sent, this is strange.");
                        }
                    }
                }
                catch (InterruptedException | ExecutionException e)
                {
                    if (e.getCause() instanceof PushNotificationException)
                    {
                        _handlePushError((PushNotificationException) e.getCause(), title, message, data, tokenToUserMap);
                    }
                    else
                    {
                        getLogger().warn("Error while sending push notification", e);
                    }
                }
            }

            List<ExpoPushMessageTicketPair<ExpoPushMessage>> zippedMessagesTickets = client.zipMessagesTickets(expoPushMessagesHandled, allTickets);

            List<ExpoPushMessageTicketPair<ExpoPushMessage>> okTicketMessages = client.filterAllSuccessfulMessages(zippedMessagesTickets);
            String okTicketMessagesString = okTicketMessages.stream().map(
                p -> "Title: " + p.message.getTitle() + ", Id:" + p.ticket.getId()
            ).collect(Collectors.joining(","));

            if (!okTicketMessages.isEmpty())
            {
                getLogger().info(
                        "Recieved OK ticket for "
                                + okTicketMessages.size()
                                + " messages: " + okTicketMessagesString
                );
            }

            List<ExpoPushMessageTicketPair<ExpoPushMessage>> errorTicketMessages = client.filterAllMessagesWithError(zippedMessagesTickets);
            String errorTicketMessagesString = errorTicketMessages.stream().map(
                p -> "Title: " + p.message.getTitle() + ", Error: " + p.ticket.getDetails().getError()
            ).collect(Collectors.joining(","));

            if (!errorTicketMessages.isEmpty())
            {
                getLogger().warn("Received ERROR push ticket for {} messages: {}", errorTicketMessages.size(), errorTicketMessagesString);

                for (ExpoPushMessageTicketPair<ExpoPushMessage> errorTicketMessage : errorTicketMessages)
                {
                    String pushToken = (String) errorTicketMessage.ticket.getDetails().getAdditionalProperties().get("expoPushToken");

                    UserIdentity user = tokenToUserMap.get(pushToken);

                    getLogger().debug("remove token {} for user {} messages: {}", pushToken, user, errorTicketMessagesString);

                    _userPreferencesHelper.removeNotificationToken(pushToken, user);
                }
            }
        }
    }

    /**
     * Handle an exception essentially when trying to push notifications to multiple expo projects, to do it again for each project
     * @param exception the exception thrown
     * @param title title of the notification
     * @param message description of the notification
     * @param data data of the notification
     * @param tokenToUserMap Reversed user/token map
     * @throws PushClientException Something went very wrong
     */
    protected void _handlePushError(PushNotificationException exception, String title, String message, Map<String, Object> data, Map<String, UserIdentity> tokenToUserMap) throws PushClientException
    {
        Map<String, Set<String>> projects = new HashMap<>();
        Exception cause = exception.exception;
        if (cause instanceof PushNotificationErrorsException)
        {
            PushNotificationErrorsException error = (PushNotificationErrorsException) cause;
            for (ExpoPushError expoPushError : error.errors)
            {
                Map<String, Object> additionalProperties = expoPushError.getAdditionalProperties();
                if (additionalProperties != null)
                {
                    Object details = additionalProperties.get("details");
                    if (details != null && details instanceof Map)
                    {
                        @SuppressWarnings("unchecked")
                        Map<String, List<String>> detailsMap = (Map<String, List<String>>) details;
                        for (Entry<String, List<String>> entry : detailsMap.entrySet())
                        {
                            if (!projects.containsKey(entry.getKey()))
                            {
                                projects.put(entry.getKey(), new HashSet<>());
                            }
                            projects.get(entry.getKey()).addAll(entry.getValue());
                        }
                    }
                }
            }
            
        }
        
        for (Entry<String, Set<String>> entry : projects.entrySet())
        {
            getLogger().warn("Trying to send push notification to multiple expo projects, one of them is {}, containing {} tokens.", entry.getKey(), entry.getValue().size());
            Map<UserIdentity, Set<String>> tokensForOneProject = new HashMap<>();
            for (String token : entry.getValue())
            {
                UserIdentity user = tokenToUserMap.get(token);
                tokensForOneProject.computeIfAbsent(user, u -> new HashSet<>()).add(token);
            }
            pushNotifications(title, message, tokensForOneProject, data);
        }
    }
}
