/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.Sort;
import org.ametys.cms.search.Sort.Order;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.content.ContentSearcherFactory.SearchModelContentSearcher;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.cms.search.ui.model.SearchUIModelExtensionPoint;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.queriesdirectory.QueryContainer;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

import com.google.common.collect.ImmutableList;

/**
 * Manager to list and execute queries
 */
public class QueriesHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = QueriesHelper.class.getName();

    /** global configuration about queries */
    public static final String QUERY_CONTAINER_CONF_ID = "plugin.mobileapp.query.container";

    /** site configuration about queries */
    public static final String QUERY_CONTAINER_SITE_CONF_ID = "mobileapp-query-container";

    /** global configuration about queries limit */
    public static final String QUERY_LIMIT_CONF_ID = "plugin.mobileapp.query.limit";

    /** The service manager */
    protected ServiceManager _manager;

    /** The Ametys object resolver */
    private AmetysObjectResolver _resolver;

    /** Queries Helper */
    private org.ametys.plugins.queriesdirectory.QueryHelper _queriesHelper;

    /** SearchUI Model Extension Point */
    private SearchUIModelExtensionPoint _searchUiEP;

    /** The helper to handler content types */
    private ContentTypesHelper _contentTypesHelper;

    /** Content Searcher Factory */
    private ContentSearcherFactory _contentSearcherFactory;

    private SiteManager _siteManager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
    }

    /**
     * Getter for AmetysObjectResolver to avoid problems at startup
     * @return AmetysObjectResolver
     */
    protected AmetysObjectResolver getResolver()
    {
        if (_resolver == null)
        {
            try
            {
                _resolver = (AmetysObjectResolver) _manager.lookup(AmetysObjectResolver.ROLE);
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while retrieving AmetysObjectResolver", e);
            }
        }
        return _resolver;
    }

    /**
     * Getter for QueriesManager to avoid problems at startup
     * @return QueriesManager
     */
    protected org.ametys.plugins.queriesdirectory.QueryHelper getQueriesManager()
    {
        if (_queriesHelper == null)
        {
            try
            {
                _queriesHelper = (org.ametys.plugins.queriesdirectory.QueryHelper) _manager.lookup(org.ametys.plugins.queriesdirectory.QueryHelper.ROLE);
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while retrieving QueriesManager", e);
            }
        }
        return _queriesHelper;
    }

    /**
     * Getter for SearchUIModelExtensionPoint to avoid problems at startup
     * @return SearchUIModelExtensionPoint
     */
    protected SearchUIModelExtensionPoint getSearchUIModelExtensionPoint()
    {
        if (_searchUiEP == null)
        {
            try
            {
                _searchUiEP = (SearchUIModelExtensionPoint) _manager.lookup(SearchUIModelExtensionPoint.ROLE);
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while retrieving SearchUIModelExtensionPoint", e);
            }
        }
        return _searchUiEP;
    }

    /**
     * Getter for ContentTypesHelper to avoid problems at startup
     * @return ContentTypesHelper
     */
    protected ContentTypesHelper getContentTypesHelper()
    {
        if (_contentTypesHelper == null)
        {
            try
            {
                _contentTypesHelper = (ContentTypesHelper) _manager.lookup(ContentTypesHelper.ROLE);
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while retrieving ContentTypesHelper", e);
            }
        }
        return _contentTypesHelper;
    }

    /**
     * Getter for SiteManager to avoid problems at startup
     * @return SiteManager
     */
    protected SiteManager getSiteManager()
    {
        if (_siteManager == null)
        {
            try
            {
                _siteManager = (SiteManager) _manager.lookup(SiteManager.ROLE);
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while retrieving SiteManager", e);
            }
        }
        return _siteManager;
    }

    /**
     * Getter for ContentSearcherFactory to avoid problems at startup
     * @return ContentSearcherFactory
     */
    protected ContentSearcherFactory getContentSearcherFactory()
    {
        if (_contentSearcherFactory == null)
        {
            try
            {
                _contentSearcherFactory = (ContentSearcherFactory) _manager.lookup(ContentSearcherFactory.ROLE);
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while retrieving SiteManager", e);
            }
        }
        return _contentSearcherFactory;
    }

    /**
     * List all queries available for the mobile app
     * @param site the site. May be null
     * @return a list of {@link Query}
     */
    public List<Query> getQueries(Site site)
    {
        String queryContainerId = site != null ? site.getValue(QUERY_CONTAINER_SITE_CONF_ID) : null;

        if (StringUtils.isBlank(queryContainerId))
        {
            queryContainerId = Config.getInstance().getValue(QUERY_CONTAINER_CONF_ID);
        }

        AmetysObject ametysObject = getResolver().resolveById(queryContainerId);
        return getQueries(ametysObject);
    }

    /**
     * Get all queries available in a specific {@link QueryContainer}
     * @param ametysObject a {@link QueryContainer} to parse, or directly a {@link Query} to return in a list
     * @return a list containing all {@link Query} available in this {@link QueryContainer}
     */
    public List<Query> getQueries(AmetysObject ametysObject)
    {
        getLogger().info("Fetch queries under query '{}'", ametysObject.getId());
        if (ametysObject instanceof QueryContainer)
        {
            QueryContainer queryContainer = (QueryContainer) ametysObject;

            return queryContainer.getChildren().stream()
                .map(q -> getQueries(q))
                .flatMap(List::stream)
                .collect(Collectors.toList());
        }
        else if (ametysObject instanceof Query)
        {
            getLogger().info("Query '{}' is not a container", ametysObject.getId());
            return ImmutableList.of((Query) ametysObject);
        }
        else
        {
            getLogger().info("Query '{}' is empty", ametysObject.getId());
            return Collections.EMPTY_LIST;
        }
    }

    /**
     * Execute a query, with a forced sort
     * @param query the query to execute
     * @param sort a list of sert elements, can contain lastValidation
     * @param checkRights check the rights while executing the query
     * @return the results of the query
     */
    public AmetysObjectIterable<Content> executeQuery(Query query, List<Sort> sort, boolean checkRights)
    {
        getLogger().info("Execute query '{}', testing rights : '{}'", query.getId(), checkRights);
        try
        {
            Long limit = Config.getInstance().getValue(QUERY_LIMIT_CONF_ID);
            Map<String, Object> exportParams = getQueriesManager().getExportParams(query);

            String model = getQueriesManager().getModelForQuery(exportParams);

            if ("solr".equals(query.getType()))
            {
                String queryStr = getQueriesManager().getQueryStringForQuery(exportParams);
                Set<String> contentTypeIds = getQueriesManager().getContentTypesForQuery(exportParams).stream().collect(Collectors.toSet());
                AmetysObjectIterable<Content> results = getContentSearcherFactory().create(contentTypeIds)
                        .setCheckRights(checkRights)
                        .withSort(sort)
                        .withLimits(0, limit.intValue())
                        .search(queryStr);
                getLogger().info("Query '{}' returned {} results", query.getId(), results.getSize());
                return results;
            }
            else if (Query.Type.SIMPLE.toString().equals(query.getType()) || Query.Type.ADVANCED.toString().equals(query.getType()))
            {
                Map<String, Object> values = getQueriesManager().getValuesForQuery(exportParams);

                Map<String, Object> contextualParameters = getQueriesManager().getContextualParametersForQuery(exportParams);
                String searchMode = getQueriesManager().getSearchModeForQuery(exportParams);
                SearchUIModel uiModel = getSearchUIModelExtensionPoint().getExtension(model);
                SearchUIModel wrappedUiModel = new SearchModelWrapper(uiModel, _manager, getLogger());

                SearchModelContentSearcher searcher = getContentSearcherFactory().create(wrappedUiModel);
                AmetysObjectIterable<Content> results = searcher
                        .setCheckRights(checkRights)
                        .withLimits(0, limit.intValue())
                        .withSort(sort)
                        .withSearchMode(searchMode)
                        .search(values, contextualParameters);
                getLogger().info("Query '{}' returned {} results", query.getId(), results.getSize());
                return results;
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error during the execution of the query '" + query.getTitle() + "' (" + query.getId() + ")", e);
        }
        return null;
    }

    /**
     * Execute all queries to return the list of the queries that return this object
     * @param content the object to test
     * @param site the site. May be null
     * @return a list containing all the impacted queries
     */
    public Set<Query> getQueriesForResult(AmetysObject content, Site site)
    {
        return getQueriesForResult(content.getId(), site);
    }

    /**
     * Execute all queries to return the list of the queries that return this object
     * @param contentId the content id
     * @param site the site. May be null
     * @return a list containing all the impacted queries
     */
    public Set<Query> getQueriesForResult(String contentId, Site site)
    {
        Map<String, Set<Query>> queriesForResult = getQueriesForResult(List.of(contentId), site);
        if (queriesForResult.containsKey(contentId))
        {
            return queriesForResult.get(contentId);
        }
        else
        {
            return Collections.EMPTY_SET;
        }
    }

    /**
     * Execute all queries to return the list of the queries that return this object
     * @param contents a list of object to test
     * @param site the site. May be null
     * @return a map containing, for each content, the list containing all the impacted queries
     */
    public Map<String, Set<Query>> getQueriesForResult(List<String> contents, Site site)
    {
        Map<String, Set<Query>> result = new HashMap<>();
        List<Query> queries = getQueries(site);

        getLogger().info("Searching for queries returning contents : {}", contents);
        for (Query query : queries)
        {
            getLogger().info("Searching with query : {}", query.getId());
            try
            {
                List<Sort> sort = getSortProperty(query, queries.size() > 1);

                AmetysObjectIterable<Content> searchResults = executeQuery(query, sort, false);
                if (searchResults != null)
                {
                    List<String> resultIds = searchResults.stream().map(c -> c.getId()).collect(Collectors.toList());

                    for (String resultId : resultIds)
                    {
                        if (contents.contains(resultId))
                        {
                            getLogger().info("Query : {} contains content : {}", query.getId(), resultId);
                            if (!result.containsKey(resultId))
                            {
                                result.put(resultId, new HashSet<>());
                            }
                            result.get(resultId).add(query);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                getLogger().error("Error during the execution of the query '" + query.getTitle() + "' (" + query.getId() + ")", e);
            }
        }
        if (result.isEmpty())
        {
            getLogger().info("No query found for contents '{}'", contents);
        }
        return result;
    }

    /**
     * Generate a list of sort properties.
     * If there are multiple queries, they have to be sorted first by date ASC. If they already are, the sort is used, otherwise a sort on lastValidation is used
     * @param query the query that will be executed
     * @param moreThanOne if false and the query already contain a sort, the original sort will be returned
     * @return a list of sort, with the first one by date if moreThanOne is true
     */
    public List<Sort> getSortProperty(Query query, boolean moreThanOne)
    {
        Map<String, Object> exportParams = getQueriesManager().getExportParams(query);
        List<Sort> baseSort = getQueriesManager().getSortForQuery(exportParams);
        if (!moreThanOne && baseSort != null && !baseSort.isEmpty())
        {
            return baseSort;
        }
        else
        {
            if (baseSort != null && !baseSort.isEmpty())
            {
                Sort firstSort = baseSort.get(0);
                Set<String> specificFields = Set.of("creationDate", "lastValidation", "lastModified");
                if (specificFields.contains(firstSort.getField()))
                {
                    firstSort.setOrder(Order.ASC);
                    return List.of(firstSort);
                }

                List<String> contentTypesForQuery = getQueriesManager().getContentTypesForQuery(exportParams);
                if (contentTypesForQuery != null)
                {
                    String[] contentTypeIdsAsArray = contentTypesForQuery.toArray(new String[contentTypesForQuery.size()]);
                    try
                    {
                        ModelItem modelItem = getContentTypesHelper().getModelItem(firstSort.getField(), contentTypeIdsAsArray, new String[0]);

                        String sortFileTypeId = modelItem.getType().getId();
                        if (ModelItemTypeConstants.DATE_TYPE_ID.equals(sortFileTypeId) || ModelItemTypeConstants.DATETIME_TYPE_ID.equals(sortFileTypeId))
                        {
                            firstSort.setOrder(Order.ASC);
                            return List.of(firstSort);
                        }
                    }
                    catch (UndefinedItemPathException e)
                    {
                        getLogger().warn("Error while fetching the model of the field '" + firstSort.getField() + "' for content types [" + String.join(", ", contentTypesForQuery) + "]", e);
                    }
                }
            }
            Sort sort = new Sort("lastValidation", Sort.Order.ASC);
            return List.of(sort);
        }
    }

    /**
     * Get a json object representing the content
     * @param content tho content to parse
     * @return a json map
     */
    public Map<String, String> getDataForContent(Content content)
    {
        Map<String, String> result = new HashMap<>();
        // feed_id
        // category_name
        // date
        // content_id
        result.put("content_id", content.getId());
        if (content.hasValue("illustration/image"))
        {
            String imgUri = "illustration/image?contentId=" + content.getId();
            String image = ResolveURIComponent.resolveCroppedImage("attribute", imgUri, PostConstants.IMAGE_SIZE, PostConstants.IMAGE_SIZE, false, true);
            result.put("image", image);
        }
        // title
        result.put("title", content.getTitle());
        // content_url
        if (content instanceof WebContent)
        {
            WebContent webContent = (WebContent) content;
            Iterator<Page> pages = webContent.getReferencingPages().iterator();
            if (pages.hasNext())
            {
                Page page = pages.next();
                String siteName = webContent.getSiteName();
                String url = getSiteManager().getSite(siteName).getUrl() + "/" + page.getSitemap().getName() + "/" + page.getPathInSitemap() + ".html";
                result.put("content_url", url);
            }

        }
        return result;
    }

    /**
     * Transform a ZonedDateTime, LocalDate or Date to an instant
     * @param o1 the date to transform
     * @param o2 a relative date, if it contains a timezone and the 1st does not, this timezone will bu used
     * @return an Instant
     */
    public Instant toInstant(Object o1, Object o2)
    {
        if (o1 instanceof ZonedDateTime)
        {
            return ((ZonedDateTime) o1).toInstant();
        }
        else if (o1 instanceof LocalDate)
        {
            if (o2 instanceof ZonedDateTime)
            {
                LocalDate ld1 = (LocalDate) o1;
                ZonedDateTime zdt2 = (ZonedDateTime) o2;
                return ld1.atStartOfDay(zdt2.getZone()).toInstant();
            }
            else
            {
                return ((LocalDate) o1).atStartOfDay(ZoneId.systemDefault()).toInstant();
            }
        }
        else if (o1 instanceof Date)
        {
            return ((Date) o1).toInstant();
        }

        return null;
    }

    /**
     * Get the date of the content, can be the date requested in the field, or by default the last validation date
     * @param content the content to read
     * @param field the field to check
     * @return the iso formatted date
     */
    public String getContentFormattedDate(Content content, String field)
    {
        String isoDate = null;
        if (StringUtils.isNotBlank(field) && content.hasValue(field))
        {
            Object value = content.getValue(field, true, null);
            if (value instanceof Date)
            {
                isoDate = DateUtils.dateToString((Date) value);
            }
            else
            {
                Instant instant = toInstant(value, null);
                if (instant != null)
                {
                    Date date = new Date(instant.toEpochMilli());
                    isoDate = DateUtils.dateToString(date);
                }
            }
        }
        else if ("lastModified".equals(field))
        {
            // lastModified is not a content field
            isoDate = DateUtils.zonedDateTimeToString(content.getLastModified());
        }

        // If no date found, use the last validation date
        if (StringUtils.isBlank(isoDate))
        {
            isoDate = DateUtils.zonedDateTimeToString(content.getLastValidationDate());
        }

        return isoDate;
    }
}
