/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.Sort;
import org.ametys.plugins.mobileapp.PostConstants;
import org.ametys.plugins.mobileapp.QueriesHelper;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Returns the list of feeds for a user
 */
public class GetFeedsContentsAction extends AbstractLoggedAction
{
    /** The Ametys object resolver */
    protected QueriesHelper _queryHelper;

    /** Authentication Token Manager */
    protected SiteManager _siteManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _queryHelper = (QueriesHelper) smanager.lookup(QueriesHelper.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
    }

    @Override
    protected Map<String, Object> doLoggedInAction(Request request, Map<String, Object> jsonParams)
    {
        Map<String, Object> result = new HashMap<>();

        String siteName = (String) getParameter(PostConstants.SITE_NAME, jsonParams, request);

        Site site = _siteManager.getSite(siteName);

        List<Query> queries = _queryHelper.getQueries(site);

        List<QuerySearchresult> results = new ArrayList<>();

        for (Query query : queries)
        {
            List<Sort> sort = _queryHelper.getSortProperty(query, queries.size() > 1);
            AmetysObjectIterable<Content> searchResults = _queryHelper.executeQuery(query, sort, true);
            if (searchResults != null)
            {
                Sort firstSort = sort.get(0);
                searchResults.stream()
                    .forEach(content -> results.add(new QuerySearchresult(query, firstSort, content)));
            }
        }

        Comparator<QuerySearchresult> comparator = new Comparator<>()
        {
            public int compare(QuerySearchresult o1, QuerySearchresult o2)
            {
                Object o1Value = getDate(o1);
                Object o2Value = getDate(o2);
                return compareDates(o1Value, o2Value);
            }

            private Object getDate(QuerySearchresult queryResult)
            {
                String field = queryResult.getSort().getField();
                Content content = queryResult.getContent();
                Object value;
                if (content.hasValue(field))
                {
                    value = content.getValue(field, true, null);
                }
                else
                {
                    value = content.getLastValidationDate();
                }
                return value;
            }
        };

        List<Map<String, String>> jsonResults;
        if (queries.size() > 1)
        {
            jsonResults = results.stream()
                .sorted(comparator)
                .map(searchResult -> contentToJson(searchResult))
                .collect(Collectors.toList());
        }
        else
        {
            jsonResults = results.stream()
                    .map(searchResult -> contentToJson(searchResult))
                    .collect(Collectors.toList());
        }


        result.put("items", jsonResults);

        return result;
    }

    /**
     * Transform a content into a json map
     * @param searchResult the search result containing tho content
     * @return a json map
     */
    protected Map<String, String> contentToJson(QuerySearchresult searchResult)
    {
        Content content = searchResult.getContent();
        Map<String, String> result = _queryHelper.getDataForContent(content);

        result.put("feed_id", searchResult.getQueryId());
        result.put("category_name", searchResult.getQueryName());

        String sortField = searchResult.getSort().getField();

        String isoDate = _queryHelper.getContentFormattedDate(content, sortField);
        result.put("date", isoDate);

        return result;
    }

    /**
     * Compare two dates (can be {@link ZonedDateTime} and/or {@link LocalDate} and/or {@link Date}
     * @param o1 a {@link ZonedDateTime} and/or {@link LocalDate} and/or {@link Date}
     * @param o2 a {@link ZonedDateTime} and/or {@link LocalDate} and/or {@link Date}
     * @return -1 if o1 is before o2, +1 if o1 is after o2, 0 if same date (not equals)
     * @throws ClassCastException one of the object is not a {@link ZonedDateTime} or {@link LocalDate} or {@link Date}
     */
    protected int compareDates(Object o1, Object o2) throws ClassCastException
    {
        if (o1 == null && o2 == null)
        {
            return 0;
        }
        else if (o1 == null)
        {
            return -1;
        }
        else if (o2 == null)
        {
            return 1;
        }
        else if (o1 instanceof Date && o2 instanceof Date)
        {
            return ((Date) o1).compareTo((Date) o2);
        }
        else if (o1 instanceof LocalDate && o2 instanceof LocalDate)
        {
            return ((LocalDate) o1).compareTo((LocalDate) o2);
        }
        else if (o1 instanceof ZonedDateTime && o2 instanceof ZonedDateTime)
        {
            return ((ZonedDateTime) o1).compareTo((ZonedDateTime) o2);
        }

        Instant i1 = _queryHelper.toInstant(o1, o2);
        Instant i2 = _queryHelper.toInstant(o2, o1);
        if (i1 != null && i2 != null)
        {
            return i1.compareTo(i2);
        }

        // One of them is not null and not a LocalDate or ZonedDateTime
        throw new ClassCastException();
    }

    /**
     * Content fetched from a query
     */
    protected class QuerySearchresult
    {
        private Query _query;
        private Sort _sort;
        private Content _content;

        /**
         * New QuerySearchresult
         * @param query the query that found this content
         * @param sort the sort used
         * @param content the content itself
         */
        public QuerySearchresult(Query query, Sort sort, Content content)
        {
            _query = query;
            _sort = sort;
            _content = content;
        }

        /**
         * The query ID
         * @return The query ID
         */
        public String getQueryId()
        {
            return _query.getId();
        }
        /**
         * The query name
         * @return The query name
         */
        public String getQueryName()
        {
            return _query.getTitle();
        }


        /**
         * The sort
         * @return The sort
         */
        public Sort getSort()
        {
            return _sort;
        }

        /**
         * The content
         * @return The content
         */
        public Content getContent()
        {
            return _content;
        }

    }

}
