/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.authentication.CredentialProvider;
import org.ametys.core.authentication.token.AuthenticationTokenManager;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.core.impl.authentication.FormCredentialProvider;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Returns the token for a user (only to be used by the mobile app on one site)
 */
public class GetTokenAction extends ServiceableAction
{
    /** Authentication Token Manager */
    protected AuthenticationTokenManager _authenticationTokenManager;

    /** The user population DAO */
    protected UserPopulationDAO _userPopulationDAO;

    /** The helper for the associations population/context */
    protected PopulationContextHelper _populationContextHelper;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** JSON Utils */
    protected JSONUtils _jsonUtils;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _authenticationTokenManager = (AuthenticationTokenManager) smanager.lookup(AuthenticationTokenManager.ROLE);

        _userPopulationDAO = (UserPopulationDAO) smanager.lookup(UserPopulationDAO.ROLE);
        _populationContextHelper = (PopulationContextHelper) smanager.lookup(PopulationContextHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);

        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);

        String body = "{}";

        if (_currentUserProvider.getUser() != null)
        {
            String generateToken = _authenticationTokenManager.generateToken(0, "mobileapp", "Token for the mobile app");

            result.put("code", 200);
            result.put("token", generateToken);
        }
        else
        {
            HttpServletRequest postReq = (HttpServletRequest) objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);
            try (InputStream postBody = postReq.getInputStream())
            {
                body = new String(postBody.readAllBytes(), StandardCharsets.UTF_8);
            }

            Map<String, Object> jsonParams = _jsonUtils.convertJsonToMap(body);

            String login;
            if (jsonParams.containsKey("login"))
            {
                login = (String) jsonParams.get("login");
            }
            else
            {
                login = request.getParameter("login");
            }

            String password;
            if (jsonParams.containsKey("password"))
            {
                password = (String) jsonParams.get("password");
            }
            else
            {
                password = request.getParameter("password");
            }
            boolean authenticated = false;

            String population = null;
            if (jsonParams.containsKey("population"))
            {
                population = (String) jsonParams.get("population");
            }

            if (StringUtils.isNotBlank(login))
            {

                UserPopulation userPopulation = _userPopulationDAO.getUserPopulation(population);

                if (userPopulation != null)
                {
                    authenticated = authenticate(login, password, request, resolver, parameters, userPopulation, userPopulation.getCredentialProviders());
                }
                else
                {
                    authenticated = authenticate(login, password, request, resolver, parameters);
                }

            }

            if (authenticated)
            {
                String generateToken = _authenticationTokenManager.generateToken(0, "mobileapp", "Token for the mobile app");

                result.put("code", 200);
                result.put("token", generateToken);
                // Do not provide the redirector. We don't want the request to be redirected
                // This shouldn't have an impact as the credential provider used to login was forced to be a FormCredentialProvider
                _currentUserProvider.logout(null);
            }
            else
            {
                result.put("code", 403);
                throw new AccessDeniedException();
            }
        }

        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }

    private boolean authenticate(String login, String password, Request request, SourceResolver resolver, Parameters parameters) throws ParameterException
    {
        String siteName = parameters.getParameter("site");
        List<String> populationContexts = List.of("/sites/" + siteName, "/sites-fo/" + siteName);
        boolean atLeastOneAvailablePopulation = false;

        for (String context : populationContexts)
        {

            Map<UserPopulation, List<CredentialProvider>> populations = _populationContextHelper.getUserPopulationsOnContexts(List.of(context), false, false).stream()
                    .map(_userPopulationDAO::getUserPopulation)
                    .collect(Collectors.toMap(Function.identity(), u -> u.getCredentialProviders()));


            for (Entry<UserPopulation, List<CredentialProvider>> entry : populations.entrySet())
            {
                atLeastOneAvailablePopulation = _tryConnect(login, password, request, resolver, context, entry.getKey(), entry.getValue()) || atLeastOneAvailablePopulation;
            }

            if (!atLeastOneAvailablePopulation)
            {
                getLogger().error("Error while logging-in from the mobile application to the '" + siteName + "' site. At least one population should be configured with a form credential provider.");
            }
        }

        return _currentUserProvider.getUser() != null;
    }

    private boolean _tryConnect(String login, String password, Request request, SourceResolver resolver, String context, UserPopulation userPopulation, List<CredentialProvider> providers)
    {
        boolean atLeastOneAvailablePopulation = false;
        for (int i = 0; i < providers.size(); i++)
        {
            CredentialProvider credentialProvider = providers.get(i);
            if (credentialProvider instanceof FormCredentialProvider)
            {
                try
                {
                    request.setAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_AUTHENTICATED, "false");
                    String loginParameters = "Username=" + URIUtils.encodeParameter(login);
                    loginParameters += "&Password=" + URIUtils.encodeParameter(password);
                    loginParameters += "&UserPopulation=" + URIUtils.encodeParameter(userPopulation.getId());
                    loginParameters += "&CredentialProviderIndex=" + i;
                    loginParameters += "&context=" + URIUtils.encodeParameter(context);

                    atLeastOneAvailablePopulation = true;

                    resolver.resolveURI("cocoon:/authenticate?" + loginParameters);
                    if (_currentUserProvider.getUser() != null)
                    {
                        break;
                    }
                }
                catch (IOException e)
                {
                    getLogger().error("Impossible to test logins on population '" + userPopulation.getId() + "' using credential provider at position '" + i + "'");
                }
            }
        }
        return atLeastOneAvailablePopulation;
    }

    private boolean authenticate(String login, String password, Request request, SourceResolver resolver, Parameters parameters, UserPopulation userPopulation, List<CredentialProvider> providers) throws ParameterException
    {
        String siteName = parameters.getParameter("site");
        List<String> populationContexts = List.of("/sites/" + siteName, "/sites-fo/" + siteName);

        for (String context : populationContexts)
        {
            _tryConnect(login, password, request, resolver, context, userPopulation, providers);
        }

        return _currentUserProvider.getUser() != null;
    }

}
