/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* Constants */
const CREDENTIALS = new javax.jcr.SimpleCredentials('ametys', []);
const ARCHIVES_SESSION = repository.login(CREDENTIALS, 'archives');
const XPATH_QUERY = "//element(*, ametys:content)[@ametys-internal:contentType = 'org.ametys.plugins.newsletter.Content.newsletter']";
const RICH_TEXT_MIGRATION_XSL_CONTENT =
"<xsl:stylesheet version=\"1.0\" xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\" xmlns=\"http://docbook.org/ns/docbook\" xmlns:docbook=\"http://docbook.org/ns/docbook\">" +
    "<xsl:template match=\"img[@data-ametys-type='local']\">" +
        "<xsl:call-template name=\"migrate-mediadata\"/>" +
    "</xsl:template>" +
    "<xsl:template match=\"media[@data-ametys-type='local']\">" +
        "<xsl:call-template name=\"migrate-mediadata\"/>" +
    "</xsl:template>" +
    "<xsl:template name=\"migrate-mediadata\">" +
        "<xsl:copy>" +
            "<xsl:copy-of select=\"@*[name() != 'href']\"/>" +
            "<xsl:attribute name=\"href\">" +
                "<xsl:choose>" +    
                    "<xsl:when test=\"contains(@href, ';')\">" +
                        "<xsl:value-of select=\"substring-after(@href, ';')\"/>" +
                    "</xsl:when>" +
                    "<xsl:otherwise>" +
                        "<xsl:value-of select=\"@href\"/>" +
                    "</xsl:otherwise>" +
                "</xsl:choose>" +
            "</xsl:attribute>" +
        "</xsl:copy>" +
    "</xsl:template>" +    
    "<xsl:template match=\"*\">" + 
        "<xsl:copy>" +
            "<xsl:copy-of select=\"@*\"/>" +
            "<xsl:apply-templates/>" +
        "</xsl:copy>" +
    "</xsl:template>" +
"</xsl:stylesheet>";

/* Imports */
const IOUtils = Java.type('org.apache.commons.io.IOUtils');
const StringUtils = Java.type("org.apache.commons.lang.StringUtils");
const outgoingReferencesExtractor = Ametys.serviceManager.lookup('org.ametys.cms.content.references.OutgoingReferencesExtractor');

/* Methods */
function __migrateRichTextsDataInAllNewsletters(session)
{
    let contents = Repository.query(XPATH_QUERY, true, session);
    let totalContents = contents.getSize();
    let handledContents = 0;
    for (let content of contents)
    {
        if (content instanceof org.ametys.cms.repository.ModifiableContent)
        {
            if (__hasToMigrateContent(content))
            {
                Content.migrate(
                    content,
                    [__migrateRichTextsDataInNewsletter.bind(null)],
                    true,
                    null,
                    false,
                    session == null
                );
            }
        }
        else
        {
            logger.warn("Unable to migrate rich-texts of the content " + content + " because it is not modifiable. This content is ignored");
        }
        
        handledContents++;
        if (handledContents % 500 == 0)
        {
            logger.info("[" + handledContents + "/" + totalContents + "] handled newsletters");
        }
    }
    
    logger.info(handledContents + " handled newsletters");
}

function __hasToMigrateContent(content)
{
    let hasToMigrate = __doesContentRichTextContainsLocalMedia(content);
    if (!hasToMigrate)
    {
        // If there is no modification in default workspace, check the live one (only if the current version is not the live one)
        let hasLiveVersion = Java.from(content.getAllLabels()).indexOf('Live') >= 0;
        let currentVersionIsLive = Java.from(content.getLabels()).indexOf('Live') >= 0;
        
        if (hasLiveVersion && !currentVersionIsLive)
        {
            content.switchToLabel('Live'); // Swith to live workspace if it exists
            hasToMigrate = __doesContentRichTextContainsLocalMedia(content);
            content.switchToRevision(null); // Reset the content to the current revision before calling the content migration function
        }
    }
    return hasToMigrate;
}

function __doesContentRichTextContainsLocalMedia(content)
{
    let richText = content.getValue("content", false, null);
    if (richText != null)
    {
        let stream = null;
        try
        {
            stream = richText.getInputStream();
            let contentByteArray = IOUtils.toByteArray(stream);
            return StringUtils.contains(new java.lang.String(contentByteArray, "UTF-8"), "data-ametys-type=\"local\"");
        }
        catch (e)
        {
            logger.error("An error occurred on content " + content.getId(), e);
            throw e;
        }
        finally
        {
            if (stream != null)
            {
                IOUtils.closeQuietly(stream);
            }
        }
    }
}

function __migrateRichTextsDataInNewsletter(content)
{
    try
    {
        let value = content.getValue("content", false, null);
        if (value != null)
        {
            let attachmentsHaveChanged = __copyAttachmentsFromOtherContents(content, value);
            __migrateRichTextData(value);
            content.setValue("content", value)
            
            logger.debug("Rich-text has been migrated for newsletter " + content);
            
            if (attachmentsHaveChanged)
            {
                // if attachments have changed, it is necessary to update the outgoing references
                let outgoingReferencesByPath = outgoingReferencesExtractor.getOutgoingReferences(content);
                content.setOutgoingReferences(outgoingReferencesByPath);
            }
        }
    }
    catch (e)
    {
        logger.error("An error occurred on content " + content.getId(), e);
        throw e;
    }
}

function __copyAttachmentsFromOtherContents(content, richText)
{
    let hasChanges = false;
    let contentOutgoingReferences = content.getOutgoingReferences().get("content");
    if (contentOutgoingReferences != null && contentOutgoingReferences.containsKey("local"))
    {
        let localContentOutgoingReferences = contentOutgoingReferences.get("local");
        for (let reference of localContentOutgoingReferences)
        {
            if (!reference.includes(content.getId() + "@"))
            {
                logger.debug("The content of newsletter '" + content.getTitle() + "' (" + content.getId() + ") has an attachment that references another content."
                            + " The script will first copy the attachment before migrate its content.");
    
                let sourceAttachment = __getAttachmentFromURI(reference, content);
                if (sourceAttachment != null)
                {
                    richText.addAttachment(sourceAttachment);
                }
                
                hasChanges = true;
            }
        }
    }
    
    return hasChanges;
}

function __getAttachmentFromURI(uri, referencingContent)
{
    // uri are like <contentId>@<attribute>;<file.ext>
    let i = uri.indexOf('@');
    let j = uri.indexOf(';', i);

    if (i > 0 && j > 0)
    {
        let contentId = uri.substring(0, i);
        let dataPath = uri.substring(i + 1, j);
        let filename = uri.substring(j + 1);
        
        try
        {
            let content = Repository.resolver.resolveById(contentId);
            let value = content.getValue(dataPath);
            if (content.isMultiple(dataPath))
            {
                for (let richText of value)
                {
                    if (richText.hasAttachment(filename))
                    {
                        return richText.getAttachment(filename);
                    }
                }
            }
            else
            {
                if (value.hasAttachment(filename))
                {
                    return value.getAttachment(filename);
                }
            }
        }
        catch (e)
        {
            logger.warn("The content of newsletter '" + referencingContent.getTitle() + "' (" + referencingContent.getId() + ") references an attachment from the nonexisting content '" + contentId + "'. Unable to copy this remote attachment");
            return null;
        }
    }
    
    // If no attachment has been found
    return null;
}

function __migrateRichTextData(richText)
{
    let inputStream = null;
    let outputStream = null;
    try
    {
        let transformer = templates.newTransformer();
        
        inputStream = richText.getInputStream();
        outputStream = richText.getOutputStream();
        
        let source = new javax.xml.transform.stream.StreamSource(inputStream);
        let result = new javax.xml.transform.stream.StreamResult(outputStream);
        
        transformer.transform(source, result);
    }
    finally
    {
        IOUtils.closeQuietly(inputStream);
        IOUtils.closeQuietly(outputStream);
    }
}

/* Execute the migration */
let saxFactory = javax.xml.transform.TransformerFactory.newInstance();
let reader = null;
let templates = null;
try
{
    /* Prepare XSL once */; 
    reader = new java.io.StringReader(RICH_TEXT_MIGRATION_XSL_CONTENT);
    let xsl = new javax.xml.transform.stream.StreamSource(reader);
    templates = saxFactory.newTemplates(xsl);
    
    logger.debug("Handling newsletter in default workspace");
    __migrateRichTextsDataInAllNewsletters(null);
    logger.debug("Handling newsletter in archives workspace");
    __migrateRichTextsDataInAllNewsletters(ARCHIVES_SESSION);
}
finally
{
    IOUtils.closeQuietly(reader);
}
