/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This UI helper provides a dialog to select one category on a categories tree.
 * See {@link #open} method.
 */
Ext.define('Ametys.cms.newsletter.helper.ChooseCategory', {
    singleton: true,
    
    /**
     * Configure and open the dialog box
     * @param {Object} config The configuration options :
     * @param {String} [config.icon] The full path to icon (16x16 pixels) for the dialog box.
     * @param {String} [config.iconCls=ametysicon-catalog2] The CSS class for the icon of the dialog box. Use only if the 'icon' configuration is undefined.
     * @param {Boolean} [config.multiple=false] `true` to allow selecting multiple tags.
     * @param {String} [config.title] The title of the dialog box.
     * @param {String} [config.value] the selected category (id).
     * @param {String} config.langContext The language context. Set to a language code such as 'fr', 'en', .. for using a specifying language.
     * @param {String} [config.defaultSiteName] The default site name.
     * @param {Function} config.callback The callback function invoked when category is selected. The callback function will received the following parameters:
     * @param {Function} config.callback.id id of selected category.
     */
    open: function (config)
    {
        config = config || {};
        
        this.langContext = config.langContext;
        this.defaultSiteName = config.defaultSiteName || Ametys.getAppParameter('siteName');
        this.multiple = config.multiple;
	    
        this._cbFn = config.callback;
        
        this.value = config.value || [];
        
        this._createDialogBox(config.iconCls || 'ametysicon-catalog2', config.icon, config.title);
        
        this._box.show();
        // Refresh the root node.
	    this._tree.getStore().load({
	    	scope: this,
	    	callback: function(records, operation, success) {
	    		// Select, expand and focus the root node
	    		var rootNode = this._tree.getRootNode();
	    		
	    		this._tree.getSelectionModel().deselectAll();
			    this._tree.getSelectionModel().select(rootNode);
			    this._tree.expandNode(rootNode);
			    this._tree.getView().focusNode(rootNode);
	    	}
	    });
    },
    
    /**
     * Creates the dialog box
     * @param {String} iconCls The CSS class for the icon of the dialog box
     * @param {String} icon The full path to icon (16x16 pixels) for the dialog box
     * @param {String} title The title of the dialog box.
     * @private
     */
    _createDialogBox: function (iconCls, icon, title)
    {
        var me = this;
        
        this._tree = Ext.create('Ametys.plugins.newsletter.NewslettersTree', {
            border: true,
            flex: 1,
            
        	langContext: this.langContext,
        	categoriesOnly: true,
		    defaultSiteName: this.defaultSiteName,
            checkMode: this.multiple,
            values: this.value
        });
        
        this._box = Ext.create('Ametys.window.DialogBox', {
            title: title || "{{i18n PLUGINS_NEWSLETTER_WIDGET_TITLE}}",
            icon: icon,
            iconCls: Ext.isEmpty(icon) ? iconCls : null,
            width: 420,
            height: 500,
            scrollable: false,
            
            bodyStyle: {
            	padding: 0
            },
            cls: 'ametys-dialogbox',
            
            layout: {
                type: 'vbox',
                align : 'stretch',
                pack  : 'start'
            },
            
            items: [{
	                xtype: 'component',
	                cls: 'text',
	                html: "{{i18n PLUGINS_NEWSLETTER_WIDGET_DESCRIPTION}}"
	            },
	            this._tree
	        ],
            
	        referenceHolder: true,
	        defaultButton: 'validate',
           
            closeAction: 'destroy',
            buttons : [{
            		reference: 'validate',
            		itemId: 'ok-btn',
            		disabled: false,
                    text :"{{i18n PLUGINS_NEWSLETTER_HELPER_CHOOSECATEGORIES_OKBUTTON}}",
                    handler : Ext.bind(this._validate, this)
                }, {
                    text :"{{i18n PLUGINS_NEWSLETTER_HELPER_CHOOSECATEGORIES_CANCELBUTTON}}",
                    handler: Ext.bind(function() {this._box.close();}, this)
                } 
            ]
        });
    },
    
    /**
     * This method updates the dialog box depending on the current selection 
     * @param {Ext.selection.Model} sm The selection model
     * @param {Ext.data.Model[]} nodes The selected records
     * @private
     */
    
    /**
     * Retrieve the current tree value, and call the callback function from the initial configuration sent to #open
     * @private
     */
    _validate: function ()
    {
        if (this.multiple)
        {

            if (this._tree.values.length == 0)
            {
                Ametys.Msg.show({
                       title: "{{i18n PLUGINS_NEWSLETTER_HELPER_CHOOSECATEGORIES_SELECTION_MULTIPLE}}",
                       msg: "{{i18n PLUGINS_NEWSLETTER_HELPER_CHOOSECATEGORIES_SELECTION_INVALID_MULTIPLE}}",
                       buttons: Ext.Msg.OK,
                       icon: Ext.MessageBox.ERROR
                });
                return;
            }
            
            this._box.close();
            if (this._cbFn)
            {
                this._cbFn(this._tree.values);
            }
        }
        else
        {
            var selected = this._tree.getSelectionModel().getSelection()[0];
            
            if (!selected || selected.get('type') != 'newsletter-category')
            {
                Ametys.Msg.show({
                    title: "{{i18n PLUGINS_NEWSLETTER_HELPER_CHOOSECATEGORIES_SELECTION_SIMPLE}}",
                    msg: "{{i18n PLUGINS_NEWSLETTER_HELPER_CHOOSECATEGORIES_SELECTION_INVALID_SIMPLE}}",
                    buttons: Ext.Msg.OK,
                    icon: Ext.MessageBox.ERROR
                });
                return;
            }
            
            this._box.close();
            if (this._cbFn)
            {
                this._cbFn(selected.get('id'));
            }
        }
    }
   
});
    