/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.auto;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;

/**
 * Automatic newsletter client side element: indicate which of the selected categories are in "automatic" mode.
 */
public class AutomaticNewsletterClientSideElement extends StaticClientSideElement
{
    
    /** The automatic newsletter extension point. */
    protected AutomaticNewsletterExtensionPoint _autoNewsletterEP;
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _autoNewsletterEP = (AutomaticNewsletterExtensionPoint) serviceManager.lookup(AutomaticNewsletterExtensionPoint.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) serviceManager.lookup(CategoryProviderExtensionPoint.ROLE);
    }
    
    /**
     * Get the parameters of the automatic newletter
     * @param categoryIds the id of categories
     * @return the automatic newletter parameters
     */
    @Callable(rights = "Plugins_Newsletter_Right_SetAutomatic")
    public Map<String, Object> getStatus(List<String> categoryIds)
    {
        Map<String, Object> params = new HashMap<>();
        
        List<Map<String, Object>> categoryWithAutoNewsletters = new ArrayList<>();
        
        for (String categoryId : categoryIds)
        {
            Category category = _categoryProviderEP.getCategory(categoryId);
            CategoryProvider provider = _categoryProviderEP.getCategoryProvider(categoryId);
            Collection<String> autoIds = provider.getAutomaticIds(categoryId);
            
            if (!autoIds.isEmpty())
            {
                Map<String, Object> categoryInfo = new HashMap<>();
                categoryInfo.put("id", categoryId);
                categoryInfo.put("title", category.getTitle());
                categoryInfo.put("count", autoIds.size());
                categoryWithAutoNewsletters.add(categoryInfo);
            }
        }
        
        params.put("categories-with-auto-newsletters", categoryWithAutoNewsletters);
        params.put("auto-newsletter-count", categoryWithAutoNewsletters.size());
        params.put("available-auto-newsletter-count", _autoNewsletterEP.getExtensionsIds().size());
        
        return params;
    }
    
}
