/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.auto;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;

/**
 * DAO for manipulating automatic newsletter categories.
 *
 */
public class AutomaticNewsletterDAO implements Serviceable, Component
{
    private CategoryProviderExtensionPoint _categoryProviderEP;
    private AutomaticNewsletterExtensionPoint _autoNewsletterEP;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _autoNewsletterEP = (AutomaticNewsletterExtensionPoint) manager.lookup(AutomaticNewsletterExtensionPoint.ROLE);
    }
    
    /**
     * Set the automatic newsletters property on a newsletter category.
     * @param categoryId The id of category
     * @param autoNewslettersIds the list of automatic newsletter ids
     * @throws ProcessingException if an error occurred when setting up the automatic newsletters
     */
    @Callable(rights = "Plugins_Newsletter_Right_SetAutomatic")
    public void setAutomaticNewsletters (String categoryId, List<String> autoNewslettersIds) throws ProcessingException
    {
        if (StringUtils.isEmpty(categoryId) || !_categoryProviderEP.hasCategory(categoryId))
        {
            throw new ProcessingException("Unknown category ID: " + categoryId);
        }
        
        CategoryProvider provider = _categoryProviderEP.getCategoryProvider(categoryId);
        
        provider.setAutomatic(categoryId, autoNewslettersIds);
    }
    
    /**
     * Get the automatic newsletters
     * @return the automatic newsletters
     */
    @Callable(rights = "Plugins_Newsletter_Right_SetAutomatic")
    public List<Map<String, Object>> getAutomaticNewsletters ()
    {
        List<Map<String, Object>> autoNewsletters = new ArrayList<>();
        
        for (String autoNewsletterId : _autoNewsletterEP.getExtensionsIds())
        {
            AutomaticNewsletter autoNewsletter = _autoNewsletterEP.getExtension(autoNewsletterId);
            
            Map<String, Object> infos = new HashMap<>();
            infos.put("id", autoNewsletterId);
            infos.put("label", autoNewsletter.getLabel());
            infos.put("description", autoNewsletter.getDescription());
            
            Map<String, Object> frequency = new HashMap<>();
            frequency.put("type", autoNewsletter.getFrequencyType().toString().toLowerCase());
            frequency.put("dayNumbers", autoNewsletter.getDayNumbers());
            infos.put("frequency", frequency);
            
            infos.put("filters", autoNewsletter.getFilters());
            
            autoNewsletters.add(infos);
        }
        
        return autoNewsletters;
    }
}
